import React from 'react';
import CrudModule from '../../common/CRUD/CrudModule';

// Inline JSON schema for the feature form.
const featureSchema = {
  title: "Feature",
  type: "object",
  required: ["name", "url", "icon", "category"],
  properties: {
    name: { type: "string", title: "Feature Name" },
    display: { type: "boolean", title: "Display" },
    url: { type: "string", title: "URL" },
    apiurl: { type: "string", title: "API URL" },
    icon: { type: "string", title: "Icon" },
    category: { type: "string", title: "Category" },
    description: { type: "string", title: "Description" }
  }
};

// Inline UI schema for the feature form.
const featureUiSchema = {
  name: { "ui:placeholder": "Enter feature name" },
  url: { "ui:placeholder": "Enter feature URL" },
  apiurl: { "ui:placeholder": "Enter API URL (optional)" },
  icon: { "ui:placeholder": "Enter icon identifier" },
  category: { "ui:placeholder": "saas or platform" }
};

const Features = () => {
  const handleCellClick = (header, item) => {
    if (header === "Feature") {
      console.log('Opening FeatureView for:', item.features_id);
    }
  };

  // Define comprehensive search fields for filtering
  const searchFields = [
    { 
      name: 'q', 
      type: 'text', 
      placeholder: 'Search features...', 
      defaultValue: '' 
    },
    { 
      name: 'category', 
      type: 'dropdown', 
      placeholder: 'Category',
      options: [
        { label: 'All Categories', value: '' },
        { label: 'SaaS', value: 'saas' },
        { label: 'Platform', value: 'platform' },
        { label: 'Other', value: 'other' },
      ]
    },
    { 
      name: 'display', 
      type: 'dropdown', 
      placeholder: 'Display Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Displayed', value: 'true' },
        { label: 'Hidden', value: 'false' },
      ]
    },
    { 
      name: 'created_date_from', 
      type: 'date', 
      placeholder: 'From Date' 
    },
    { 
      name: 'created_date_to', 
      type: 'date', 
      placeholder: 'To Date' 
    },
  ];

  return (
    <CrudModule
      collectionName="features"
      title=""
      description="Manage system features/modules."
      headerMapping={{
        "Feature": { 
          key: "name", 
          label: "Feature",
          clickable: true,
          sortable: true,
          formatter: (value) => value || '-'
        },
        "Display": { 
          key: "display",
          label: "Display",
          sortable: true,
          formatter: (value) => value ? "Yes" : "No" 
        },
        "URL": { 
          key: "url", 
          label: "URL",
          sortable: true,
          clickable: false 
        },
        "API URL": { 
          key: "apiurl", 
          label: "API URL",
          sortable: true,
          clickable: false 
        },
        "Icon": { 
          key: "icon", 
          label: "Icon",
          sortable: true,
          clickable: false 
        },
        "Category": { 
          key: "category", 
          label: "Category",
          sortable: true,
          clickable: false 
        },
      }}
      dataKey="items"
      identifierKey="features_id"
      schema={featureSchema}
      uiSchema={featureUiSchema}
      fetchUrl="/crud/features/"
      searchFields={searchFields} // Pass the searchFields here
      rowsPerPage={10}
      actionFormats={[
        { label: "view", modal: true, url: "/features/view/{features_id}" },
        { label: "edit", modal: true, url: "/features/update/{features_id}" },
        { label: "delete", modal: true, url: "/features/delete/{features_id}" },
      ]}
      onCellClick={handleCellClick}
    />
  );
};

export default Features;