import React, { useState } from 'react';
import CrudModule from '../common/CRUD/CrudModule';

const pricingTypeSchema = {
  title: "Pricing Type",
  type: "object",
  required: ["pricing_type"],
  properties: {
    pricing_type: { 
      type: "string", 
      title: "Pricing Type",
      style: { width: '250px' } // Added width
    },
  }
};

const pricingTypeUiSchema = {
  pricing_type: { 
    "ui:placeholder": "Enter pricing type",
    "ui:options": {
      style: { width: '650px' } // Added width
    }
  },
  status: {
    "ui:widget": "select",
    "ui:options": {
      style: { width: '250px' }, // Added width
      enumOptions: [
        { value: "active", label: "Active" },
        { value: "inactive", label: "Inactive" },
        { value: "archived", label: "Archived" }
      ]
    }
  }
};

const PricingType = () => {
  const collectionName = "pricing_types";
  const rowsPerPage = 10;

  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: '',
    status: ''
  });

  const formatDate = (dateString) => {
    if (!dateString) return '—';
    try {
      return new Date(dateString).toLocaleString();
    } catch (e) {
      return dateString;
    }
  };

  const buildFetchUrl = () => {
    const query = new URLSearchParams();
    query.append('skip', searchParams.skip);
    query.append('limit', searchParams.limit);
    if (searchParams.q) query.append('q', searchParams.q);
    if (searchParams.status) query.append('status', searchParams.status);
    return `/crud/${collectionName}/?${query.toString()}`;
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  const styles = {
    searchInput: {
      width: '250px',
      padding: '8px 12px',
      borderRadius: '4px',
      border: '1px solid #ccc'
    }
  };

  return (
    <CrudModule
      collectionName={collectionName}
      title="Pricing Types"
      description="Manage pricing types in the system."
      headerMapping={{
        "Pricing Type": {
          key: "pricing_type",
          default: "—"
        },
        // "Status": { // ✅ Added Status header
        //   key: "status",
        //   default: "—"
        // },
        "Created Date": {
          key: "created_date",
          render: (value) => formatDate(value)
        }
      }}
      dataKey="items"
      identifierKey="pricing_types_id"
      schema={pricingTypeSchema}
      uiSchema={pricingTypeUiSchema}
      fetchUrl={buildFetchUrl()}
      searchFields={[
        {
          name: 'q',
          type: 'text',
          placeholder: 'Search pricing types...',
          defaultValue: searchParams.q,
          style: styles.searchInput
        },
        {
          name: 'status',
          type: 'dropdown',
          placeholder: 'Status',
          options: [
            { label: 'All', value: '' },
            { label: 'Active', value: 'active' },
            { label: 'Inactive', value: 'inactive' },
            { label: 'Archived', value: 'archived' }
          ],
          defaultValue: searchParams.status,
          style: { width: '250px' }
        }
      ]}
      rowsPerPage={rowsPerPage}
      actionFormats={[
        { label: "view", modal: true, url: `/pricing-types/view/{pricing_types_id}` },
        { label: "edit", modal: true, url: `/pricing-types/update/{pricing_types_id}` },
        { label: "delete", modal: true, url: `/pricing-types/delete/{pricing_types_id}` },
      ]}
      onSearch={handleSearch}
      onPageChange={handlePageChange}
      emptyMessage="No pricing types found."
    />
  );
};

export default PricingType;
