
import React, { useEffect, useState } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import AccountList from '../Accounts/AccountsList';

const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'activities', label: 'Activities', icon: 'history' },
  { id: 'info', label: 'Info', icon: 'info' },
  { id: 'accounts', label: 'Accounts', icon: 'account_balance' },
  { id: 'users', label: 'Users', icon: 'people' }
];

const PartnerView = ({ partnerId, onClose }) => {
  const [activeTab, setActiveTab] = useState('info');
  const [partnerData, setPartnerData] = useState(null);
  const [comments, setComments] = useState([]);
  const [logs, setLogs] = useState([]);
  const [newComment, setNewComment] = useState('');
  const [isPosting, setIsPosting] = useState(false);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [workforceData, setWorkforceData] = useState([]);
  const [workforceLoading, setWorkforceLoading] = useState(false);
  const [workforceError, setWorkforceError] = useState('');

  useEffect(() => {
    async function fetchData() {
      try {
        const [partnerRes, commentsRes, logsRes] = await Promise.all([
          axios.get(`/partners/${partnerId}`),
          axios.get(`/cx/crm/comments/${partnerId}`),
          axios.get(`/cx/crm/logs/${partnerId}`)
        ]);
        setPartnerData(partnerRes.data);
        setComments(commentsRes.data.comments || []);
        setLogs(logsRes.data.logs || []);
      } catch (err) {
        setError('Error loading partner details.');
      } finally {
        setLoading(false);
      }
    }
    fetchData();
  }, [partnerId]);

  useEffect(() => {
    const fetchWorkforceData = async () => {
      if (activeTab === 'users' && partnerId) {
        setWorkforceLoading(true);
        try {
          const response = await axios.get('/workforce/list/', {
            params: {
              referred_account_id: partnerId, // Changed to referred_account_id
              limit: 100
            }
          });
          setWorkforceData(response.data.workforce || []);
        } catch (err) {
          setWorkforceError('Failed to load workforce data.');
          console.error('Error fetching workforce:', err);
        } finally {
          setWorkforceLoading(false);
        }
      }
    };

    fetchWorkforceData();
  }, [activeTab, partnerId]);

  const postComment = async () => {
    if (!newComment.trim()) return;
    setIsPosting(true);
    try {
      const response = await axios.post(`/cx/crm/comment/`, {
        partner_id: partnerId,
        comment: newComment
      });
      setComments([response.data, ...comments]);
      setNewComment('');
    } catch {
      alert('Failed to post comment.');
    } finally {
      setIsPosting(false);
    }
  };

  const Icon = ({ name, color = '#000', style = {} }) => (
    <span
      className="material-symbols-outlined"
      style={{ fontSize: '24px', color, verticalAlign: 'middle', marginRight: '6px', ...style }}
    >
      {name}
    </span>
  );

  // Simple workforce table component
  const WorkforceTable = ({ workforce }) => {
    return (
      <div style={{ overflowX: 'auto' }}>
        <table style={{ width: '100%', borderCollapse: 'collapse' }}>
          <thead>
            <tr style={{ backgroundColor: '#f8f9fa' }}>
              <th style={{ padding: '12px 15px', textAlign: 'left', borderBottom: '1px solid #e0e0e0' }}>Name</th>
              <th style={{ padding: '12px 15px', textAlign: 'left', borderBottom: '1px solid #e0e0e0' }}>Email</th>
              <th style={{ padding: '12px 15px', textAlign: 'left', borderBottom: '1px solid #e0e0e0' }}>Role</th>
              <th style={{ padding: '12px 15px', textAlign: 'left', borderBottom: '1px solid #e0e0e0' }}>Status</th>
            </tr>
          </thead>
          <tbody>
            {workforce.length > 0 ? (
              workforce.map((worker) => (
                <tr key={worker.workforce_id} style={{ borderBottom: '1px solid #eee' }}>
                  <td style={{ padding: '12px 15px' }}>{worker.first_name} {worker.last_name}</td>
                  <td style={{ padding: '12px 15px' }}>{worker.email_id || 'N/A'}</td>
                  <td style={{ padding: '12px 15px' }}>{worker.workforce_category_name || 'N/A'}</td>
                  <td style={{ padding: '12px 15px' }}>
                    <span style={{ 
                      backgroundColor: worker.status === 'active' ? '#4caf50' : '#9e9e9e',
                      color: '#fff',
                      padding: '4px 10px',
                      borderRadius: '12px',
                      fontSize: '0.8em',
                      display: 'inline-block'
                    }}>
                      {worker.status === 'active' ? "Active" : "Inactive"}
                    </span>
                  </td>
                </tr>
              ))
            ) : (
              <tr>
                <td colSpan="4" style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
                  No workforce found
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>
    );
  };

  const renderTabContent = () => {
    if (loading) {
      return (
        <div style={{ padding: '20px', textAlign: 'center' }}>
          Loading partner data...
        </div>
      );
    }

    if (error) {
      return (
        <div style={{ padding: '20px', color: '#f44336' }}>
          {error}
        </div>
      );
    }

    switch (activeTab) {
      case 'info':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px', marginBottom: '20px' }}>
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Partner Information</h5>
                <p><strong>Partner Name:</strong> {partnerData.partner_name}</p>
                <p><strong>Title:</strong> {partnerData.title}</p>
                <p><strong>Industry:</strong> {partnerData.industry}</p>
                <p><strong>Status:</strong> {partnerData.is_active ? "Active" : "Inactive"}</p>
              </div>
              
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Contact Information</h5>
                <p><strong>Name:</strong> {partnerData.name}</p>
                <p><strong>Email:</strong> {partnerData.email}</p>
                <p><strong>Mobile:</strong> {partnerData.mobile}</p>
              </div>
            </div>

            <div style={{ marginBottom: '20px' }}>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Address</h5>
              <div style={{ backgroundColor: '#f8f9fa', padding: '15px', borderRadius: '8px' }}>
                <p><strong>Address:</strong> {partnerData.address}</p>
                <p><strong>City:</strong> {partnerData.city}</p>
                <p><strong>State:</strong> {partnerData.state}</p>
                <p><strong>Country:</strong> {partnerData.country}</p>
                <p><strong>Postal Code:</strong> {partnerData.postal_code}</p>
              </div>
            </div>
          </div>
        );

      case 'accounts':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ 
              display: 'flex', 
              justifyContent: 'space-between', 
              alignItems: 'center',
              marginBottom: '20px'
            }}>
              <h5>Accounts Referred by {partnerData?.partner_name}</h5>
            </div>

            <div style={{ 
              border: '1px solid #e0e0e0', 
              borderRadius: '8px',
              overflow: 'hidden'
            }}>
              <AccountList 
                initialSearchQuery="" 
                rowsPerPage={10}
                referredAccountId={partnerId} // Changed to referredAccountId
                isEmbedded={true}
                isGlobalAccess={true}
              />
            </div>
          </div>
        );

      case 'users':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ 
              display: 'flex', 
              justifyContent: 'space-between', 
              alignItems: 'center',
              marginBottom: '20px'
            }}>
              <h5>Workforce Referred by {partnerData?.partner_name}</h5>
              <span style={{ 
                padding: '4px 12px', 
                backgroundColor: '#e8f5e9', 
                borderRadius: '12px',
                fontSize: '0.9em',
                color: '#2e7d32'
              }}>
                Total: {workforceData.length}
              </span>
            </div>

            {workforceLoading ? (
              <div style={{ padding: '40px', textAlign: 'center' }}>
                Loading workforce...
              </div>
            ) : workforceError ? (
              <div style={{ padding: '20px', color: '#f44336', textAlign: 'center' }}>
                {workforceError}
              </div>
            ) : (
              <div style={{ 
                border: '1px solid #e0e0e0', 
                borderRadius: '8px',
                overflow: 'hidden',
                padding: '15px',
                backgroundColor: '#fff'
              }}>
                <WorkforceTable workforce={workforceData} />
              </div>
            )}
          </div>
        );

      case 'activities':
        return (
          <div style={{ padding: '20px' }}>
            <h5>Activity Logs</h5>
            <div style={{ maxHeight: '400px', overflowY: 'auto' }}>
              {logs.length > 0 ? (
                logs.map(log => (
                  <div key={log.id} style={{ 
                    borderBottom: '1px solid #eee', 
                    marginBottom: '10px', 
                    paddingBottom: '10px' 
                  }}>
                    <p style={{ margin: 0, fontWeight: 'bold' }}>
                      {log.activity} - 
                      <span style={{ fontWeight: 'normal', fontSize: '0.9em', marginLeft: '8px' }}>
                        {new Date(log.timestamp).toLocaleString()}
                      </span>
                    </p>
                    <p style={{ margin: '5px 0 0 0' }}>{log.details}</p>
                  </div>
                ))
                ) : (
                <p style={{ color: '#666', textAlign: 'center' }}>No activity logs found</p>
              )}
            </div>
          </div>
        );

      case 'dashboard':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ 
              display: 'grid', 
              gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', 
              gap: '15px',
              marginBottom: '20px',
              padding: '15px',
              backgroundColor: '#f8f9fa',
              borderRadius: '8px'
            }}>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '2rem', fontWeight: 'bold'}}>
                  {comments.length}
                </div>
                <div>Comments</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '2rem', fontWeight: 'bold' }}>
                  {logs.length}
                </div>
                <div>Activities</div>
              </div>
            </div>
            
            <div style={{ 
              marginBottom: '20px', 
              padding: '15px', 
              border: '1px solid #e0e0e0', 
              borderRadius: '8px'
            }}>
              <h5>Add Comment</h5>
              <textarea
                value={newComment}
                onChange={(e) => setNewComment(e.target.value)}
                placeholder="Add a comment..."
                rows={3}
                style={{ 
                  width: '100%', 
                  padding: '10px',
                  border: '1px solid #ddd',
                  borderRadius: '4px',
                  marginBottom: '10px'
                }}
              />
              <button 
                onClick={postComment} 
                disabled={isPosting}
                style={{
                  padding: '8px 16px',
                  backgroundColor: '#34c759',
                  color: 'white',
                  border: 'none',
                  borderRadius: '4px',
                  cursor: 'pointer'
                }}
              >
                {isPosting ? 'Posting...' : 'Post Comment'}
              </button>
            </div>
            
            <div>
              <h5>Recent Comments</h5>
              <div style={{ maxHeight: '200px', overflowY: 'auto' }}>
                {comments.length > 0 ? (
                  comments.slice(0, 5).map(comment => (
                    <div key={comment.id} style={{ 
                      borderBottom: '1px solid #eee', 
                      marginBottom: '10px', 
                      paddingBottom: '10px' 
                    }}>
                      <p style={{ margin: 0, fontWeight: 'bold' }}>
                        {comment.team_member_id} - 
                        <span style={{ fontWeight: 'normal', fontSize: '0.9em', marginLeft: '8px' }}>
                          {new Date(comment.created_date).toLocaleString()}
                        </span>
                      </p>
                      <p style={{ margin: '5px 0 0 0' }}>{comment.comment}</p>
                    </div>
                  ))
                ) : (
                  <p style={{ color: '#666', textAlign: 'center' }}>No comments yet</p>
                )}
              </div>
            </div>
          </div>
        );

      default:
        return (
          <div style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
            Content for {TAB_CONFIG.find(tab => tab.id === activeTab)?.label} tab is coming soon.
          </div>
        );
    }
  };

  return (
    <Modal onClose={onClose}>
      <div style={{
        display: 'flex',
        alignItems: 'center',
        padding: '10px',
        justifyContent: 'space-between',
        borderBottom: '1px solid #eee'
      }}>
        <div style={{ display: 'flex', alignItems: 'center' }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold',
            marginRight: '16px'
          }}>
            {partnerData?.partner_name?.charAt(0)?.toUpperCase() || 'P'}
          </div>
          <div>
            <div style={{ fontWeight: '600', fontSize: '20px', color: 'black' }}>
              {partnerData?.partner_name || 'Partner'}
            </div>
            <div style={{ fontSize: '14px', color: '#666', marginTop: '4px' }}>
              {partnerData?.industry || 'No industry specified'}
            </div>
          </div>
        </div>
      </div>

      <div style={{
        display: 'flex',
        borderBottom: '1px solid #e0e0e0',
        marginBottom: '0',
        padding: '0 10px',
        overflowX: 'auto',
        scrollbarWidth: 'none',
        msOverflowStyle: 'none'
      }}>
        {TAB_CONFIG.map(tab => (
          <button
            key={tab.id}
            style={{
              padding: '12px 16px',
              border: 'none',
              backgroundColor: 'transparent',
              cursor: 'pointer',
              fontSize: '14px',
              fontWeight: activeTab === tab.id ? '600' : '400',
              color: activeTab === tab.id ? '#34c759' : '#666',
              position: 'relative',
              display: 'flex',
              alignItems: 'center',
              gap: '8px',
              whiteSpace: 'nowrap',
              flexShrink: 0
            }}
            onClick={() => setActiveTab(tab.id)}
          >
            <Icon name={tab.icon} color={activeTab === tab.id ? '#34c759' : '#666'} />
            {tab.label}
            {activeTab === tab.id && (
              <div style={{
                position: 'absolute',
                bottom: '-1px',
                left: '0',
                right: '0',
                height: '2px',
                backgroundColor: '#34c759'
              }} />
            )}
          </button>
        ))}
      </div>

      <div style={{ 
        maxHeight: 'calc(100vh - 200px)', 
        overflowY: 'auto',
        padding: '0'
      }}>
        {renderTabContent()}
      </div>
    </Modal>
  );
};

export default PartnerView;