import React, { useState, useEffect } from 'react';
import axios from 'axios';
import PartnerForm from './PartnerForm';
import Modal from '../../Layouts/Modal';

const PartnerUpdate = ({ partnerId, onClose, onUpdateSuccess }) => {
  const [partnerData, setPartnerData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');

  useEffect(() => {
    const fetchPartnerData = async () => {
      try {
        const res = await axios.get(`/partners/${partnerId}`);
        const data = res.data;
        
        const statusMap = {
          0: "Trial",
          1: "Active",
          2: "Unpaid",
          3: "Expired",
          4: "Cancelled",
          5: "Suspended",
          6: "Paused"
        };

        // Convert numeric status to string if needed
        if (data.subscription_status !== undefined) {
          if (typeof data.subscription_status === "number") {
            data.subscription_status = statusMap[data.subscription_status] || "Trial";
          }
        }

        setPartnerData(data);
        setLoading(false);
      } catch (err) {
        setError('Error fetching partner data.');
        setLoading(false);
        console.error('Fetch error:', err.response?.data || err.message);
      }
    };

    fetchPartnerData();
  }, [partnerId]);

  const handleSubmit = async (formData) => {
    try {
      // Prepare the data for submission
      const dataToSubmit = {
        ...formData,
        // Ensure we don't send undefined values
        subscription_status: formData.subscription_status || "Trial"
      };

      // Remove any undefined or null values
      Object.keys(dataToSubmit).forEach(key => {
        if (dataToSubmit[key] === undefined || dataToSubmit[key] === null) {
          delete dataToSubmit[key];
        }
      });

      const res = await axios.post(`/partners/${partnerId}`, dataToSubmit, {
        headers: {
          'Content-Type': 'application/json'
        }
      });

      alert("Partner updated successfully!");
      if (onUpdateSuccess) onUpdateSuccess(res.data);
      onClose();
    } catch (err) {
      console.error('Update error:', err.response?.data || err.message);
      alert(`Error updating partner: ${err.response?.data?.message || err.message}`);
    }
  };

  return (
    <Modal title="Edit Partner" onClose={onClose}>
      {loading ? (
        <div>Loading partner data...</div>
      ) : error ? (
        <div className='error'>{error}</div>
      ) : (
        <PartnerForm 
          initialData={partnerData}
          onSubmit={handleSubmit}
        />
      )}
    </Modal>
  );
};

export default PartnerUpdate;