

import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';

import PartnerView from './PartnerView';
import PartnerUpdate from './PartnerUpdate';
import PartnerDB from  './PartnerDB';

const PARTNERS_HEADER_TO_KEY_MAP = {
  "Partner Title": { 
    key: "partner_name", 
    label: "Partner Title", 
    clickable: true,
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Name": { 
    key: "name", 
    label: "Name", 
    clickable: true,
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Email": { 
    key: "email", 
    label: "Email", 
    clickable: false,
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Type": { 
    key: "partner_type", 
    label: "Type", 
    clickable: false,
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Date": { 
    key: "created_date", 
    label: "Date", 
    clickable: false,
    sortable: false,
    formatter: (date) => date ? new Date(date).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    }) : '-'
  },
  "Status": { 
    key: "is_active", 
    label: "Status", 
    clickable: false,
    sortable: false,
    formatter: (active) => (
      <span style={{ color: active ? 'green' : 'red' }}>
        {active ? 'Active' : 'Inactive'}
      </span>
    )
  },
};

const PartnersList = ({ initialSearchQuery = '', rowsPerPage = 10 }) => {
  const fetchDataUrl = `/partners/`;
  
  // selectedModal holds an object { partnerId, modalType } where modalType can be "view", "update", or "delete"
  const [selectedModal, setSelectedModal] = useState(null);
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  const searchFields = [
    { 
      name: 'created_date_from', 
      type: 'date', 
      placeholder: 'From Date' 
    },
    { 
      name: 'created_date_to', 
      type: 'date', 
      placeholder: 'To Date' 
    },
    { 
      name: 'q', 
      type: 'text', 
      placeholder: 'Search partners...', 
      defaultValue: initialSearchQuery 
    },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'Active', value: 'active' },
        { label: 'Inactive', value: 'inactive' }
      ]
    },
   
  ];

  const transformPartnerData = (apiData) => {
    if (!apiData || !apiData.partners) {
      console.error('Invalid API data structure', apiData);
      return { total_count: 0, partners: [] };
    }
    
    return apiData;
  };

  // Handle popup actions – open the proper modal based on the action label.
  const handleActionClick = (partnerId, action) => {
    if (action.label === "view") {
      setSelectedModal({ partnerId, modalType: "view" });
    } else if (action.label === "edit") {
      setSelectedModal({ partnerId, modalType: "update" });
    } else if (action.label === "delete" || action.label === "block") {
      setSelectedModal({ partnerId, modalType: "delete" });
    }
  };

  const handleCellClick = (header, item) => {
    const mappingValue = PARTNERS_HEADER_TO_KEY_MAP[header];
    console.log('Cell Clicked:', header, item);

    if (!mappingValue) {
      console.warn('No mapping found for header:', header);
      return;
    }

    if (mappingValue.clickable) {
      console.log('Opening PartnerView for:', item.partner_id);
      setSelectedModal({
        partnerId: item.partner_id,
        modalType: 'view'
      });
    }
  };

  const closeModal = () => setSelectedModal(null);

  const refreshList = () => {
    console.log('Refreshing partner list');
    setRefreshTrigger(prev => prev + 1);
  };

  const actionFormats = [
    { label: "view", modal: true, url: "/partners/view/{id}" },
    { label: "delete", modal: true, url: "/partners/delete/{id}" },
    { label: "block", modal: true, url: "/partners/update/{id}" },
    { label: "edit", modal: true, url: "/partners/update/{id}" },
  ];

  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={PARTNERS_HEADER_TO_KEY_MAP}
        dataKey="partners"
        identifierKey="partner_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        refreshTrigger={refreshTrigger}
        dataTransformer={transformPartnerData}
        searchFields={searchFields}
        renderCell={(value, column) => {
          if (value === undefined || value === null) {
            return <span style={{ color: '#999' }}>None</span>;
          }
          return String(value);
        }}
        tableProps={{
          striped: true,
          hover: true,
          responsive: true,
          style: { marginTop: '1rem' }
        }}
      />

      {selectedModal && selectedModal.modalType === "view" && (
        <PartnerView 
          partnerId={selectedModal.partnerId} 
          onClose={closeModal} 
        />
      )}

      {selectedModal && selectedModal.modalType === "update" && (
        <PartnerUpdate 
          partnerId={selectedModal.partnerId} 
          onClose={closeModal} 
          onUpdateSuccess={refreshList}
        />
      )}

      {selectedModal && selectedModal.modalType === "delete" && (
        <PartnerDB 
          partnerId={selectedModal.partnerId} 
          onClose={closeModal} 
          onDeleteSuccess={refreshList}
        />
      )}
    </div>
  );
};

export default PartnersList;
