import React, { useState, useEffect } from 'react';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';

const Form = withTheme(MaterialUITheme);

export const partnerSchema = {
  type: "object",
  required: ["partner_name", "title", "email"],
  properties: {
    partner_name: { type: "string", title: "Partner Name" },
    title: { type: "string", title: "Title" },
    industry: { type: "string", title: "Industry" },
    name: { type: "string", title: "Name" },
    email: { type: "string", title: "Email" },
    mobile: { type: "string", title: "Mobile" },
    address: { type: "string", title: "Address" },
    city: { type: "string", title: "City" },
    state: { type: "string", title: "State" },
    country: { type: "string", title: "Country" },
    postal_code: { type: "string", title: "Postal Code" },
    is_active: { type: "boolean", title: "Active", default: true },
    partner_type: {
      type: "integer",
      title: "Partner Type",
      enum: [0, 1, 2, 3],
      enumNames: ["Free", "Startup", "Premium", "Enterprise"],
      default: 0
    },
    subscription_status: { 
      type: "string", 
      title: "Subscription Status",
      enum: ["Trial", "Active", "Unpaid", "Expired", "Cancelled", "Suspended", "Paused"],
      default: "Trial"
    }
  }
};

export const uiSchema = {
  is_active: {
    "ui:widget": "checkbox"
  },
  subscription_status: {
    "ui:placeholder": "Select subscription status"
  }
};

const CustomFormTemplate = ({ children, onSubmit }) => {
  return (
    <form onSubmit={onSubmit} className="partner-form-container">
      {children}
    </form>
  );
};

const PartnerForm = ({ initialData, onSubmit }) => {
  const [formData, setFormData] = useState({});
  const [statusMessage, setStatusMessage] = useState('');
  const [errors, setErrors] = useState([]);

  useEffect(() => {
    const statusMap = {
      0: "Trial",
      1: "Active",
      2: "Unpaid",
      3: "Expired",
      4: "Cancelled",
      5: "Suspended",
      6: "Paused"
    };

    let updatedData = initialData ? { ...initialData } : {};

    if (updatedData.subscription_status !== undefined) {
      if (typeof updatedData.subscription_status === "number") {
        updatedData.subscription_status = statusMap[updatedData.subscription_status] || "Trial";
      } else if (!partnerSchema.properties.subscription_status.enum.includes(updatedData.subscription_status)) {
        updatedData.subscription_status = "Trial";
      }
    }

    setFormData(updatedData);
  }, [initialData]);

  const handleSubmit = async ({ formData }) => {
    setErrors([]);
    try {
      await onSubmit(formData);
      setStatusMessage('Partner saved successfully!');
    } catch (error) {
      setStatusMessage('Error saving Partner.');
      console.error('Submission error:', error);
    }
  };

  const handleErrors = (formErrors) => {
    setErrors(formErrors.map(error => error.stack));
  };

  return (
    <div className="partner-form-wrapper">
      {statusMessage && <p className="status-message">{statusMessage}</p>}
      {errors.length > 0 && (
        <div className="error-messages">
          {errors.map((error, index) => (
            <p key={index} className="error-message">{error}</p>
          ))}
        </div>
      )}
      <Form
        schema={partnerSchema}
        uiSchema={uiSchema}
        formData={formData}
        onChange={({ formData }) => setFormData(formData)}
        onSubmit={handleSubmit}
        onError={handleErrors}
        validator={validator}
        liveValidate={false}
        noHtml5Validate={true}
        FormTemplate={CustomFormTemplate}
      />
    </div>
  );
};

export default PartnerForm;