// src/components/adminpages/Invoices/InvoiceView.js
import React, { useEffect, useState } from "react";
import axios from "axios";
import Modal from "../../Layouts/Modal";

// --- Shared UI bits (mirrors SubscriptionView look) ---
const TAB_CONFIG = [
  { id: "overview", label: "Overview", icon: "dashboard" },
  { id: "account",  label: "Account",  icon: "apartment" },
  { id: "payment",  label: "Payment",  icon: "payments"  },
];

const Icon = ({ name, color = "#34c759" }) => (
  <span className="material-symbols-outlined" style={{ fontSize: 20, color, verticalAlign: "middle" }}>
    {name}
  </span>
);

const Header = ({ title, subtitle }) => (
  <div style={{ display: "flex", alignItems: "center", padding: "12px 16px", borderBottom: "1px solid #eee", gap: 12 }}>
    <div style={{
      width: 40, height: 40, borderRadius: 999, background: "#eee",
      display: "flex", alignItems: "center", justifyContent: "center",
      fontWeight: 700, color: "#333"
    }}>
      {(title || "I").slice(0,1).toUpperCase()}
    </div>
    <div>
      <div style={{ fontWeight: 600, fontSize: 20, color: "#000" }}>{title || "Invoice"}</div>
      {subtitle ? <div style={{ fontSize: 13, color: "#6b7280", marginTop: 2 }}>{subtitle}</div> : null}
    </div>
  </div>
);

const TabBar = ({ tabs, active, onChange }) => (
  <div style={{ display: "flex", borderBottom: "1px solid #e0e0e0", padding: "0 8px", overflowX: "auto" }}>
    {tabs.map(t => (
      <button
        key={t.id}
        onClick={() => onChange(t.id)}
        style={{
          padding: "10px 14px", background: "transparent", border: "none", cursor: "pointer",
          color: active === t.id ? "#34c759" : "#666", fontWeight: active === t.id ? 600 : 400, display: "flex", gap: 8
        }}
      >
        <Icon name={t.icon} color={active === t.id ? "#34c759" : "#666"} />
        {t.label}
        {active === t.id && <div style={{ height: 2, background: "#34c759", position: "absolute", bottom: 0 }} />}
      </button>
    ))}
  </div>
);

const Row = ({ label, value }) => (
  <div style={{ marginBottom: 8 }}>
    <strong>{label}:</strong> {value ?? "—"}
  </div>
);

const Card = ({ title, children }) => (
  <div style={{ background: "#f8f9fa", border: "1px solid #e6e8eb", borderRadius: 12, padding: 16 }}>
    {title ? <h4 style={{ margin: 0, marginBottom: 10, fontWeight: 700 }}>{title}</h4> : null}
    {children}
  </div>
);

// --- Main ---
const InvoiceView = ({ invoiceId, onClose }) => {
  const [invoice, setInvoice] = useState(null);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState("");
  const [active, setActive] = useState("overview");

  // lazy account detail (only when account tab is opened)
  const [acct, setAcct] = useState(null);
  const [acctLoading, setAcctLoading] = useState(false);
  const [acctErr, setAcctErr] = useState("");

  useEffect(() => {
    (async () => {
      try {
        setLoading(true); setErr("");
        const { data } = await axios.get(`/invoices/${invoiceId}`);
        setInvoice(data?.invoice || data);
      } catch (e) {
        setErr("Failed to load invoice.");
      } finally {
        setLoading(false);
      }
    })();
  }, [invoiceId]);

  // Derive account and invoices from the invoice payload; no extra API calls
  const derivedAcct = React.useMemo(() => (
    invoice?.account || invoice?.account_details || invoice?.accountInfo || null
  ), [invoice]);
  const derivedAcctInvoices = React.useMemo(() => (
    invoice?.account_invoices || invoice?.invoices || invoice?.account?.invoices || []
  ), [invoice]);


  useEffect(() => {
    const loadAccount = async () => {
      if (active !== "account") return;
      if (!invoice?.account_id || acct || acctLoading) return;
      try {
        setAcctLoading(true); setAcctErr("");
        const { data } = await axios.get(`/accounts/${invoice.account_id}`);
        setAcct(data);
      } catch (e) {
        setAcctErr("Failed to load account info.");
      } finally {
        setAcctLoading(false);
      }
    };
    loadAccount();
  }, [active, invoice, acct, acctLoading]);

  const Overview = () => (
    <div style={{ padding: 16 }}>
      <div style={{ display: "grid", gap: 16, gridTemplateColumns: "repeat(auto-fit, minmax(280px, 1fr))" }}>
        <Card title="Invoice Details">
          <Row label="Status" value={invoice?.status} />
          <Row label="Amount" value={invoice ? `${invoice.amount ?? "—"} ${invoice.currency ?? ""}` : "—"} />
          <Row label="Account ID" value={invoice?.account_id} />
          {invoice?.invoice_description && (
            <div style={{ marginTop: 10 }}>
              <div style={{ fontWeight: 600, marginBottom: 6 }}>Description</div>
              <div style={{ whiteSpace: "pre-wrap" }}>{invoice.invoice_description}</div>
            </div>
          )}
        </Card>
        <Card title="Dates">
          <Row label="Invoice Date" value={invoice?.invoice_date} />
          <Row label="Due Date" value={invoice?.due_date} />
          <Row label="Created" value={invoice?.created_date ? new Date(invoice.created_date).toLocaleString() : "—"} />
        </Card>
      </div>
    </div>
  );

  
const Account = () => (
  <div style={{ padding: 16 }}>
    <Card title={(derivedAcct?.account_name ?? derivedAcct?.name ?? "—")}>
      <Row label="Email" value={derivedAcct?.email} />
      <Row label="Mobile" value={derivedAcct?.mobile} />
      <Row label="Website" value={derivedAcct?.website} />
      <Row label="Industry" value={derivedAcct?.industry} />
    </Card>
    <div style={{ height: 16 }} />
    <div style={{ display: "grid", gap: 16, gridTemplateColumns: "repeat(auto-fit, minmax(220px, 1fr))" }}>
      <Card title="Address">
        <Row label="Street" value={derivedAcct?.address?.street} />
        <Row label="City" value={derivedAcct?.address?.city} />
        <Row label="State" value={derivedAcct?.address?.state} />
        <Row label="Country" value={derivedAcct?.address?.country} />
        <Row label="Postal" value={derivedAcct?.address?.postal_code} />
      </Card>
      <Card title="Workforce">
        <Row label="Total" value={derivedAcct?.total_workforce ?? derivedAcct?.workforce_count ?? 0} />
        <Row label="Limit" value={derivedAcct?.workforce_count ?? "—"} />
      </Card>
      <Card title="Fleet">
        <Row label="Total" value={derivedAcct?.total_fleets ?? derivedAcct?.fleet_count ?? 0} />
        <Row label="Limit" value={derivedAcct?.fleet_count ?? "—"} />
      </Card>
    </div>

    <div style={{ height: 20 }} />
    <Card title="Invoices (from invoice payload)">
      {Array.isArray(derivedAcctInvoices) && derivedAcctInvoices.length > 0 ? (
        <div style={{ display: "grid", gap: 8 }}>
          {derivedAcctInvoices.map((iv, idx) => (
            <div key={(iv.invoice_id || iv.id || idx)} style={{ padding: 10, border: "1px solid #eee", borderRadius: 8 }}>
              <div style={{ fontWeight: 600 }}>{iv.invoice_title || iv.title || "Invoice"}</div>
              <div style={{ fontSize: 13, color: "#6b7280" }}>
                {(iv.amount != null ? `${iv.amount} ${iv.currency || ""}` : "—")} · {iv.status || "—"}
              </div>
            </div>
          ))}
        </div>
      ) : (
        <div>No invoices present on the invoice payload.</div>
      )}
    </Card>
  </div>
);

  const Payment = () => (
    <div style={{ padding: 16 }}>
      <div style={{ display: "grid", gap: 16, gridTemplateColumns: "repeat(auto-fit, minmax(280px, 1fr))" }}>
        <Card title="Payment">
          <Row label="Amount" value={invoice ? `${invoice.amount ?? "—"} ${invoice.currency ?? ""}` : "—"} />
          <Row label="Payment Status" value={invoice?.payment_status ?? "—"} />
          <Row label="Method" value={invoice?.payment_method ?? "—"} />
          <Row label="Transaction Ref" value={invoice?.transaction_id ?? "—"} />
          <Row label="Last Paid" value={invoice?.paid_at ?? "—"} />
        </Card>
        <Card title="Subscription">
          <Row label="Status" value={invoice?.subscription_status ?? "—"} />
          <Row label="Type" value={invoice?.subscription_type ?? "—"} />
          <Row label="Billing Period" value={invoice?.billing_period ?? "—"} />
          <Row label="Billing Cycle" value={invoice?.billing_cycle ?? "—"} />
          <Row label="Subscription ID" value={invoice?.subscription_id ?? "—"} />
        </Card>
      </div>
    </div>
  );

  const headerTitle = invoice?.invoice_title || "Invoice";
  const headerSubtitle = invoice?.status ? `Status: ${invoice.status}` : "";

  return (
    <Modal onClose={onClose}>
      <Header title={headerTitle} subtitle={headerSubtitle} />
      <TabBar tabs={TAB_CONFIG} active={active} onChange={setActive} />

      {loading ? (
        <div style={{ padding: 16 }}>Loading…</div>
      ) : err ? (
        <div style={{ padding: 16, color: "crimson" }}>{err}</div>
      ) : !invoice ? (
        <div style={{ padding: 16 }}>Not found.</div>
      ) : (
        <div style={{ maxHeight: "calc(100vh - 200px)", overflowY: "auto" }}>
          {active === "overview" && <Overview />}
          {active === "account"  && <Account />}
          {active === "payment"  && <Payment />}
        </div>
      )}
    </Modal>
  );
};

export default InvoiceView;
