import React, { useEffect, useState, useRef } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import InvoiceForm from './InvoiceForm';

const InvoiceUpdate = ({ invoiceId, onClose, onUpdateSuccess }) => {
  const [initial, setInitial] = useState(null);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState('');
  const [ok, setOk] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);

  const fetchedRef = useRef(false);

  useEffect(() => {
    if (!invoiceId) return;
    if (fetchedRef.current) return;
    fetchedRef.current = true;

    const controller = new AbortController();
    const load = async () => {
      try {
        setLoading(true); setErr('');
        const res = await axios.get(`/invoices/${invoiceId}`, { signal: controller.signal });
        const inv = res?.data?.invoice || res?.data || {};
        setInitial({
          account_id: inv.account_id ?? '',
          active_subscription_id: inv.active_subscription_id ?? '',
          invoice_type: inv.invoice_type ?? 'Setup',
          invoice_title: inv.invoice_title ?? '',
          invoice_description: inv.invoice_description ?? '',
          amount: inv.amount ?? '',
          currency: inv.currency ?? 'USD',
          status: inv.status ?? 'New',
          invoice_date: inv.invoice_date ?? '',
          due_date: inv.due_date ?? '',
          user_id: inv.user_id ?? '',
        });
      } catch (e) {
        if (axios.isCancel?.(e) || e?.name === 'CanceledError') return;
        console.error(e);
        setErr('Error fetching invoice.');
      } finally {
        setLoading(false);
      }
    };
    load();

    return () => controller.abort();
  }, [invoiceId]);

  const handleSubmit = async (formData) => {
    if (isSubmitting) return;
    setIsSubmitting(true);
    setOk(''); setErr('');
    try {
      // Prepare the data for submission - similar to PartnerUpdate
      const dataToSubmit = {
        ...formData,
        // Ensure we don't send undefined values
        invoice_id: invoiceId
      };

      // Remove any undefined or null values
      Object.keys(dataToSubmit).forEach(key => {
        if (dataToSubmit[key] === undefined || dataToSubmit[key] === null) {
          delete dataToSubmit[key];
        }
      });

      console.log('Submitting invoice data:', dataToSubmit);

      const res = await axios.post(`/invoices/${invoiceId}`, dataToSubmit, {
        headers: {
          'Content-Type': 'application/json'
        }
      });

      setOk('Invoice updated successfully!');
      onUpdateSuccess && onUpdateSuccess(res.data);
      setTimeout(() => onClose && onClose(), 1200);
    } catch (e) {
      console.error('Update error:', e.response?.data || e.message);
      setErr(`Error updating invoice: ${e.response?.data?.message || e.message}`);
      setIsSubmitting(false);
    }
  };

  return (
    <Modal title="Edit Invoice" onClose={onClose}>
      {loading ? (
        <div style={{ padding: 20 }}>Loading…</div>
      ) : err ? (
        <div style={{ padding: 20, color: '#f44336' }}>{err}</div>
      ) : (
        <div style={{ padding: 20 }}>
          {ok && <div style={{ marginBottom: 12, color: '#10b981' }}>{ok}</div>}
          <InvoiceForm 
            onSubmit={handleSubmit} 
            isUpdateForm 
            initialData={initial} 
            isSubmitting={isSubmitting}
          />
        </div>
      )}
    </Modal>
  );
};

export default InvoiceUpdate;