import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import InvoiceView from './InvoiceView';
import InvoiceUpdate from './InvoiceUpdate';
import InvoiceDB from './InvoiceDB';

const HEADER_TO_KEY = {
  Title:          { key: 'invoice_title',  label: 'Title',        sortable: true,  clickable: true },
  Amount:         { key: 'amount',         label: 'Amount',       sortable: true, formatter: (v) => (v != null ? `$${Number(v).toFixed(2)}` : '-') },
  Currency:       { key: 'currency',       label: 'Currency',     sortable: false },
  Status:         { key: 'status',         label: 'Status',       sortable: true },
  Type:           { key: 'invoice_type',   label: 'Invoice Type', sortable: true },
  'Invoice Date': { key: 'invoice_date',   label: 'Invoice Date', sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
  'Due Date':     { key: 'due_date',       label: 'Due Date',     sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
  'Created Date':   { key: 'created_date',   label: 'Created On',   sortable: true, formatter: (d) => d ? new Date(d).toLocaleString() : '-' },
};

const InvoiceList = ({ initialSearchQuery = '', rowsPerPage = 10, accountId, isEmbedded = false }) => {
  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: initialSearchQuery,
    status: '',
    created_date_from: '',
    created_date_to: '',
  });

  const [selected, setSelected] = useState(null); // { invoiceId, modalType }

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'status',
      type: 'dropdown',
      placeholder: 'Status',
      options: [
        { label: 'All', value: '' },
        { label: 'New', value: 'New' },
        { label: 'Draft', value: 'Draft' },
        { label: 'Sent', value: 'Sent' },
        { label: 'Paid', value: 'Paid' },
        { label: 'Overdue', value: 'Overdue' },
        { label: 'Cancelled', value: 'Cancelled' },
        { label: 'Unpaid', value: 'Unpaid' },
      ],
    },
    { name: 'created_date_from', type: 'date', placeholder: 'From Date' },
    { name: 'created_date_to', type: 'date', placeholder: 'To Date' },
  ];

  const actionFormats = [
    { label: 'view',   modal: true, url: '/invoice/view/{id}' },
    { label: 'edit',   modal: true, url: '/invoice/update/{id}' },
    { label: 'delete', modal: true, url: '/invoice/delete/{id}' },
  ];

  const onActionClick = (invoiceId, action) => {
    if (['view', 'edit', 'delete'].includes(action.label)) {
      setSelected({ invoiceId, modalType: action.label });
    }
  };

  const onCellClick = (header, item) => {
    if (header === 'Title') {
      const invoiceId = item.invoice_id || item.id;
      if (invoiceId) setSelected({ invoiceId, modalType: 'view' });
    }
  };

  const onSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      created_date_from: params.created_date_from || '',
      created_date_to: params.created_date_to || '',
      skip: 0,
    }));
  };

  const onPageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage,
    }));
  };

  const buildUrl = () => {
    // Use the provided accountId (for embedded view) or fall back to localStorage
    const acc = accountId || localStorage.getItem('account_id') || '';
    let url = `/invoices/?account_id=${acc}&skip=${searchParams.skip}&limit=${searchParams.limit}`;
    if (searchParams.q) url += `&q=${encodeURIComponent(searchParams.q)}`;
    if (searchParams.status) url += `&status=${encodeURIComponent(searchParams.status)}`;
    if (searchParams.created_date_from) url += `&created_date_from=${searchParams.created_date_from}`;
    if (searchParams.created_date_to)   url += `&created_date_to=${searchParams.created_date_to}`;
    return url;
  };

  return (
    <div>
      <SearchPage
        fetchDataUrl={buildUrl()}
        ListComponent={List}
        itemKeyMapping={HEADER_TO_KEY}
        dataKey="invoices"
        identifierKey="invoice_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={onActionClick}
        onCellClick={onCellClick}
        searchFields={searchFields}
        onSearch={onSearch}
        onPageChange={onPageChange}
      />

      {selected?.modalType === 'view' && (
        <InvoiceView invoiceId={selected.invoiceId} onClose={() => setSelected(null)} />
      )}
      {selected?.modalType === 'edit' && (
        <InvoiceUpdate invoiceId={selected.invoiceId} onClose={() => setSelected(null)} onUpdateSuccess={() => {}} />
      )}
      {selected?.modalType === 'delete' && (
        <InvoiceDB invoiceId={selected.invoiceId} onClose={() => setSelected(null)} onDeleteSuccess={() => {}} />
      )}
    </div>
  );
};

export default InvoiceList;