import React, { useEffect } from 'react';
import Form from '@rjsf/core';
import validator from '@rjsf/validator-ajv8';
import AutoCompleteWidget from '../../common/Builder/AutoCompleteWidget';

const today = new Date().toISOString().split('T')[0];
const dueDate = new Date(Date.now() + 7*24*60*60*1000).toISOString().split('T')[0];

const STATUS_ENUM = ['New','Draft','Sent','Paid','Overdue','Cancelled','Unpaid'];

const fullSchema = {
  title: 'Invoice',
  type: 'object',
  required: ['account_id','invoice_title','amount','currency','status','invoice_date','due_date','invoice_type'],
  properties: {
    account_id: { type: 'string', title: 'Account' },
    invoice_title: { type: 'string', title: 'Invoice Title' },
    invoice_description: { type: 'string', title: 'Description' },
    amount: { type: 'number', title: 'Amount' },
    currency: { type: 'string', title: 'Currency', enum: ['USD','INR','EUR'], default: 'USD' },
    status: { type: 'string', title: 'Status', enum: STATUS_ENUM, default: 'New' },
    invoice_type: { type: 'string', title: 'Invoice Type', enum: ['Setup','Subscription','Development','Support','Misc'], default: 'Setup' },
    invoice_date: { type: 'string', format: 'date', title: 'Invoice Date', default: today },
    due_date: { type: 'string', format: 'date', title: 'Due Date', default: dueDate },
    user_id: { type: 'string', title: 'User ID', default: '' }
  }
};

const baseSchema = {
  title: 'Update Invoice',
  type: 'object',
  required: ['status'],
  properties: {
    status: { type: 'string', title: 'Status', enum: STATUS_ENUM, default: 'New' },
    invoice_description: { type: 'string', title: 'Internal Note', default: '' }
  }
};

const uiSchema = {
  account_id: {
    'ui:widget': AutoCompleteWidget,
    'ui:options': {
      api: '/accounts/',
      listKey: 'accounts',
      searchField: 'q',
      labelField: 'account_name',
      valueField: 'account_id',
      debounce: 300,
      placeholder: 'Search account…'
    }
  },
  invoice_description: { 'ui:widget': 'textarea', 'ui:options': { rows: 3 } },
  amount: { 'ui:widget': 'text', 'ui:placeholder': 'Amount' },
  currency: { 'ui:widget': 'select' },
  status: { 'ui:widget': 'select' },
  invoice_type: { 'ui:widget': 'select' },
  invoice_date: { 'ui:widget': 'date' },
  due_date: { 'ui:widget': 'date' },
  user_id: { 'ui:widget': 'hidden' }
};

const transformErrors = () => [];

const InvoiceForm = ({ initialData, onSubmit, isUpdateForm = false }) => {
  useEffect(() => {
    const style = document.createElement('style');
    style.innerHTML = `
      .boxed-input input, .boxed-input select, .boxed-input textarea {
        border: 1px solid #ccc; border-radius: 4px; padding: 10px; background:#fff; width:100%;
      }
      .boxed-input input:focus, .boxed-input select:focus, .boxed-input textarea:focus { border-color:#000; outline: none; }
      button[type="submit"] { background:#f0f0f0; color:#00c853; padding:10px 20px; border:1px solid #fff; border-radius:4px; font-weight:700; }
      button[type="submit"]:hover { background:#00c853; color:#fff; border-color:#00c853; }
    `;
    document.head.appendChild(style);
    return () => document.head.removeChild(style);
  }, []);

  const schema = fullSchema; // Always use full schema so Edit has all fields

  return (
    <div style={{ maxWidth: 720 }}>
      <Form
        schema={schema}
        uiSchema={uiSchema}
        formData={initialData || {
          user_id: localStorage.getItem('user_id') || '',
          account_id: localStorage.getItem('account_id') || ''
        }}
        validator={validator}
        transformErrors={transformErrors}
        showErrorList={false}
        onSubmit={({ formData }) => onSubmit(formData)}
        liveValidate
      />
    </div>
  );
};

export default InvoiceForm;
