import React, { useEffect, useState } from 'react';
import axios from 'axios';
import { useParams, useNavigate } from 'react-router-dom';
import '../../themes/blue/Command.css';
import FormBuilder from '../../common/Builder/FormBuilder';

const NewApp = ({ EditAppId }) => {
    const [categories, setCategories] = useState([]);
    const [defaultValues, setDefaultValues] = useState(null);
    const [showSuccess, setShowSuccess] = useState(false);
    const roles = parseInt(localStorage.getItem("roles")) || 3;
    const { app_param_id } = useParams();
    const history = useNavigate();
    const app_id = app_param_id || EditAppId;

    console.log(EditAppId + " " + app_id);

    useEffect(() => {

        if (app_id) {
            fetchAppData(app_id);
        }
        fetchCategoriesData();
    }, [app_id]);

    const fetchAppData = async (id) => {
        try {
            const response = await axios.get(`/appflow/a/${id}`);
            const data = response.data;
            // Properly format or default the values as needed for each field
            setDefaultValues({
                name: data.name || '',
                category_id: data.category_id || '',
                call_type: data.call_type || '',
                api_schema: data.api_schema || '',
                auth: data.auth || '',
                icon: data.icon || '',
                description: data.description || '',
                approot: data.approot || '',
                // Ensure status is set to boolean as expected by checkbox field
                status: data.status
            });
    
            } catch (error) {
            console.error("Error fetching app data:", error);
        }
    };

    const fetchCategoriesData = async () => {
        try {
            const response = await axios.get('/appflow/categories');
            setCategories(response.data);
        } catch (error) {
            console.error("Error fetching categories data:", error);
        }
    };

    const handleFinalSubmission = async (formData) => {
        const userId = localStorage.getItem('user_id');
        const accountId = localStorage.getItem('account_id');
        const appData = {
            ...formData,
            user_id: userId,
            call_type: formData.call_type,
            status: formData.status,
            account_id: roles === 1 ? "" : accountId
        };

        const payload = new FormData();
        payload.append('app', JSON.stringify(appData));

        const url = app_id ? `/appflow/${app_id}` : '/appflow/';
        const method = app_id ? 'put' : 'post';

        try {
            const response = await axios({ method, url, data: payload, headers: { 'Content-Type': 'multipart/form-data' } });
            if (response.status === 200) setShowSuccess(true);
        } catch (error) {
            console.error('Error submitting the form:', error);
        }
    };

    const resetForm = () => {
        setShowSuccess(false);
        setDefaultValues(null);
        //history('/a/apps/new');
    };

    const config = {
        formType: 'chat',
        steps: [
            {
                fields: [
                    { name: 'name', type: 'text', label: 'Name', placeholder: 'Name of the app', mandatory: true, minlength: 3 },
                    { name: 'category_id', type: 'select', label: 'App Category', placeholder: 'App Category', options: categories.map(category => ({ value: category.id, label: category.name })), mandatory: true },
                    { name: 'call_type', type: 'select', label: 'Call Type', placeholder: 'Select Type', options: [{ value: 'openapi', label: 'Open API Standards' }, { value: 'sdk', label: 'SDK' }], mandatory: true },
                    { name: 'api_schema', type: 'textarea', label: 'API Schema (json)', rows: 20, placeholder: 'OpenAPI Schema - For all the functional end points, params & responses', mandatory: true, minlength: 100 },
                    { name: 'auth', type: 'select', label: 'Authentication Mode', placeholder: 'Select Mode', options: [{ value: 'no_auth', label: 'None' }, { value: 'apikey', label: 'API Key' }, { value: 'oauth', label: 'OAuth' }, { value: 'other', label: 'Other' }], mandatory: true },
                    { name: 'icon', type: 'url', label: 'Icon', placeholder: 'App icon URL', mandatory: true },
                    { name: 'description', type: 'text', label: 'Description (Optional)', placeholder: 'Description of app', mandatory: true, minlength: 10 },
                    { name: 'approot', type: 'text', label: 'SDK URL (Optional)', placeholder: 'Enter the app root (only for SDK)', mandatory: false },
                    { name: 'status', type: 'checkbox', label: 'Active', checked: false }
                ],
                defaultValues: defaultValues
            }
        ]
    };

    return (
        <div className="FormContainer">
            {showSuccess ? (
                <div>
                    <p>{app_id ? 'App updated successfully!' : 'App added successfully!'}</p>
                    <button onClick={resetForm}>Add another app</button>
                </div>
            ) : (
                <FormBuilder config={config} onFinalSubmit={handleFinalSubmission} onCancel={null} />
            )}
        </div>
    );
};

export default NewApp;
