import React, { useState } from 'react';
import CrudModule from '../common/CRUD/CrudModule';

const addonsSchema = {
  title: "Addons",
  type: "object",
  required: ["name"],
  properties: {
    name: { 
      type: "string", 
      title: "Addons Name",
      style: { width: '250px' } // Inline style for form input
    },
  }
};

const addonsUiSchema = {
  name: { 
    "ui:placeholder": "Enter addons name",
    "ui:options": {
      style: { width: '650px' } // Additional style for the input
    }
  },
  status: {
    "ui:widget": "select",
    "ui:options": {
      enumOptions: [
        { value: "active", label: "Active" },
        { value: "inactive", label: "Inactive" },
        { value: "archived", label: "Archived" }
      ]
    }
  }
};

const Addons = () => {
  const collectionName = "addons";
  const rowsPerPage = 10;

  const [searchParams, setSearchParams] = useState({
    skip: 0,
    limit: rowsPerPage,
    q: '',
    status: ''
  });

  const formatDate = (dateString) => {
    if (!dateString) return '—';
    try {
      return new Date(dateString).toLocaleString();
    } catch (e) {
      return dateString;
    }
  };

  const buildFetchUrl = () => {
    const query = new URLSearchParams();
    query.append('skip', searchParams.skip);
    query.append('limit', searchParams.limit);
    if (searchParams.q) query.append('q', searchParams.q);
    if (searchParams.status) query.append('status', searchParams.status);
    return `/crud/${collectionName}/?${query.toString()}`;
  };

  const handleSearch = (params) => {
    setSearchParams(prev => ({
      ...prev,
      q: params.q || '',
      status: params.status || '',
      skip: 0
    }));
  };

  const handlePageChange = (page) => {
    setSearchParams(prev => ({
      ...prev,
      skip: (page - 1) * rowsPerPage
    }));
  };

  // Styles object
const styles = {
  container: {
    width: '100%',          // Take full width
    maxWidth: '1500px',
    margin: 0,              // Remove auto margins
    padding: '0 0px',      // Add some padding only on sides
    boxSizing: 'border-box' // Ensure padding is included in width
  },
  searchInput: {
    width: '300px',
    padding: '8px 12px',
    borderRadius: '4px',
    border: '1px solid #ccc',
    marginRight: '10px'
  },
  searchContainer: {
    display: 'flex',
    alignItems: 'center',
    marginBottom: '20px',
  },
};

  return (
    <div style={styles.container}>
      <CrudModule
        collectionName={collectionName}
        title="Addons"
        description="Manage addons in the system."
        headerMapping={{
          "Addons Name": {
            key: "name",
            default: "—"
          },
          "Created Date": {
            key: "created_date",
            render: (value) => formatDate(value)
          }
        }}
        dataKey="items"
        identifierKey="addons_id"
        schema={addonsSchema}
        uiSchema={addonsUiSchema}
        fetchUrl={buildFetchUrl()}
        searchFields={[
          {
            name: 'q',
            type: 'text',
            placeholder: 'Search addons...',
            defaultValue: searchParams.q,
            style: styles.searchInput // Apply inline style
          },
          {
            name: 'status',
            type: 'dropdown',
            placeholder: 'Status',
            options: [
              { label: 'All', value: '' },
              { label: 'Active', value: 'active' },
              { label: 'Inactive', value: 'inactive' },
              { label: 'Archived', value: 'archived' }
            ],
            defaultValue: searchParams.status,
            style: { width: '250px' } // Inline style for dropdown
          }
        ]}
        rowsPerPage={rowsPerPage}
        actionFormats={[
          { label: "view", modal: true, url: "/addons/view/{addons_id}" },
          { label: "edit", modal: true, url: "/addons/update/{addons_id}" },
          { label: "delete", modal: true, url: "/addons/delete/{addons_id}" },
        ]}
        onSearch={handleSearch}
        onPageChange={handlePageChange}
        emptyMessage="No addons found."
        customRowRenderer={(row) => ({
          ...row,
          created_date: formatDate(row.created_date)
        })}
      />
    </div>
  );
};

export default Addons;