import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { 
  LineChart, Line, BarChart, Bar,
  XAxis, YAxis, CartesianGrid, Tooltip, 
  Legend, ResponsiveContainer, Cell 
} from 'recharts';

const Dashboard = () => {
    const name = localStorage.getItem('name') || 'User';
    const roles = localStorage.getItem('roles') || '1';
    const navigate = useNavigate();
    const [dashboardData, setDashboardData] = useState(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState(null);
    const [dateRange, setDateRange] = useState({
        start: '',
        end: ''
    });

    // Mock data for charts
    const usageTrends = [
        { name: 'Jan', value: 400 },
        { name: 'Feb', value: 0 },
        { name: 'Mar', value: 600 },
        { name: 'Apr', value: 800 },
        { name: 'May', value: 500 },
        { name: 'Jun', value: 900 }
    ];

    // Heatmap data using Recharts
    const heatmapData = [
        { name: 'Q1', value: 10 },
        { name: 'Q2', value: 20 },
        { name: 'Q3', value: 30 },
        { name: 'Q4', value: 40 },
        { name: 'Q5', value: 50 },
        { name: 'Q6', value: 60 }
    ];

    useEffect(() => {
        if (roles !== '1') {
            navigate('/logout');
        } else {
            fetchDashboardData();
        }
    }, [dateRange, roles, navigate]);

    const fetchDashboardData = async () => {
        try {
            setLoading(true);
            const response = await axios.get('dashboard/metrics', {
                params: {
                    skip: 0,
                    limit: 10,
                    created_date_from: dateRange.start || undefined,
                    created_date_to: dateRange.end || undefined
                }
            });
            setDashboardData(response.data);
            setLoading(false);
        } catch (err) {
            setError(err.message);
            setLoading(false);
        }
    };

    const handleDateChange = (e, type) => {
        setDateRange(prev => ({
            ...prev,
            [type]: e.target.value
        }));
    };

    if (loading) {
        return (
            <div style={{ 
                display: 'flex', 
                flexDirection: 'column', 
                justifyContent: 'center', 
                alignItems: 'center', 
                height: '100vh',
                background: 'white'
            }}>
                <div className="loader" style={{
                    border: '5px solid #f3f3f3',
                    borderTop: '5px solid #3498db',
                    borderRadius: '50%',
                    width: '50px',
                    height: '50px',
                    animation: 'spin 1s linear infinite'
                }}></div>
                <p style={{ marginTop: '20px', color: '#555' }}>Loading dashboard data...</p>
                <style>{`
                    @keyframes spin {
                        0% { transform: rotate(0deg); }
                        100% { transform: rotate(360deg); }
                    }
                `}</style>
            </div>
        );
    }

    if (error) {
        return (
            <div style={{
                display: 'flex',
                flexDirection: 'column',
                justifyContent: 'center',
                alignItems: 'center',
                height: '100vh',
                padding: '20px',
                textAlign: 'center',
                background: 'white'
            }}>
                <h3 style={{ color: '#e53e3e' }}>Error Loading Dashboard</h3>
                <p style={{ margin: '15px 0', color: '#718096' }}>{error}</p>
                <button 
                    onClick={fetchDashboardData}
                    style={{
                        padding: '10px 20px',
                        backgroundColor: '#38a169',
                        color: 'white',
                        border: 'none',
                        borderRadius: '5px',
                        cursor: 'pointer',
                        fontSize: '1rem',
                        fontWeight: 'bold'
                    }}
                >
                    Retry
                </button>
            </div>
        );
    }

    if (!dashboardData) {
        return (
            <div style={{
                display: 'flex',
                flexDirection: 'column',
                justifyContent: 'center',
                alignItems: 'center',
                height: '100vh',
                padding: '20px',
                textAlign: 'center',
                background: 'white'
            }}>
                <h3 style={{ color: '#3182ce' }}>No Data Available</h3>
                <p style={{ marginTop: '10px', color: '#718096' }}>There is no dashboard data to display</p>
            </div>
        );
    }

    return (
        <div style={{ 
            padding: '20px', 
            background: 'white',
            minHeight: '100vh'
        }}>
            {/* Header Section */}
            <div style={{ 
                display: 'flex', 
                flexWrap: 'wrap',
                justifyContent: 'space-between', 
                alignItems: 'center', 
                marginBottom: '25px',
                gap: '20px',
               
            }}>
                <h2 style={{ 
                    margin: 0,
                    fontSize: '1.8rem',
                    fontWeight: '700',
                    color: '#2d3748'
                }}></h2>
                
                <div style={{ 
                    display: 'flex', 
                    flexWrap: 'wrap',
                    alignItems: 'center', 
                    gap: '15px'
                }}>
                    <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                        <label style={{ fontWeight: '500', color: '#4a5568' }}>From:</label>
                        <input 
                            type="date" 
                            value={dateRange.start || dashboardData.date_range?.start?.substring(0, 10) || ''}
                            onChange={(e) => handleDateChange(e, 'start')}
                            style={{ 
                                padding: '8px 12px',
                                border: '1px solid #e2e8f0',
                                borderRadius: '6px',
                                fontSize: '0.95rem'
                            }}
                        />
                    </div>
                    <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                        <label style={{ fontWeight: '500', color: '#4a5568' }}>To:</label>
                        <input 
                            type="date" 
                            value={dateRange.end || dashboardData.date_range?.end?.substring(0, 10) || ''}
                            onChange={(e) => handleDateChange(e, 'end')}
                            style={{ 
                                padding: '8px 12px',
                                border: '1px solid #e2e8f0',
                                borderRadius: '6px',
                                fontSize: '0.95rem'
                            }}
                        />
                    </div>
                    <button 
                        onClick={fetchDashboardData} 
                        style={{ 
                            padding: '8px 20px',
                            backgroundColor: '#38a169',
                            color: 'white',
                            border: 'none',
                            borderRadius: '6px',
                            cursor: 'pointer',
                            fontWeight: '600',
                            fontSize: '0.95rem',
                            transition: 'background-color 0.2s',
                            display: 'flex',
                            alignItems: 'center',
                            gap: '8px'
                        }}
                    >
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16">
                            <path d="M8 3a5 5 0 1 0 4.546 2.914.5.5 0 0 1 .908-.417A6 6 0 1 1 8 2v1z"/>
                            <path d="M8 4.466V.534a.25.25 0 0 1 .41-.192l2.36 1.966c.12.1.12.284 0 .384L8.41 4.658A.25.25 0 0 1 8 4.466z"/>
                        </svg>
                        Refresh
                    </button>
                </div>
            </div>

            {/* Metrics Grid */}
            <div style={{
                display: 'grid',
                gridTemplateColumns: 'repeat(auto-fit, minmax(240px, 1fr))',
                gap: '18px',
                marginBottom: '30px'
            }}>
                {[
                    { title: 'Total Users', value: dashboardData.overview.users_count},
                    { title: 'Active Subscriptions', value: dashboardData.overview.active_subscriptions},
                    { title: 'Active Projects', value: dashboardData.overview.active_projects},
                    { title: 'Total Revenue', value: `$${dashboardData.overview.total_revenue.toFixed(2)}` },
                    { title: 'Workforce', value: dashboardData.workforce_fleet_trends.workforce_count},
                    { title: 'Fleets', value: dashboardData.workforce_fleet_trends.fleets_count},
                    { title: 'Vendors', value: dashboardData.workforce_fleet_trends.vendors_count},
                    { title: 'System Users', value: dashboardData.workforce_fleet_trends.users_count}
                    //trendColor: "#52c41a"
                ].map((metric, index) => (
                    <div 
                        key={index} 
                        style={{
                            backgroundColor: 'white',
                            padding: '20px 15px',
                            borderRadius: '10px',
                            boxShadow: '0 2px 8px rgba(0,0,0,0.06)',
                            textAlign: 'center',
                            transition: 'transform 0.2s',
                            cursor: 'pointer',
                            borderLeft: `4px solid ${metric.trendColor}`,
                            display: 'flex',
                            flexDirection: 'column',
                            justifyContent: 'center',
                            alignItems: 'center'
                        }}
                    >
                        <h1 style={{ 
                            margin: '10px 0', 
                            fontSize: '2rem', 
                            fontWeight: '700',
                            color: '#1a202c',
                            textAlign: 'center'
                        }}>
                            {metric.value}
                        </h1>
                        <p style={{ 
                            margin: '8px 0', 
                            fontWeight: '600',
                            color: '#2d3748',
                            fontSize: '1.05rem',
                            textAlign: 'center'
                        }}>
                            {metric.title}
                        </p>
                        <small style={{ 
                            color: metric.trendColor,
                            fontWeight: '500',
                            fontSize: '0.9rem',
                            textAlign: 'center'
                        }}>
                            {metric.trend}
                        </small>
                    </div>
                ))}
            </div>

            {/* Charts Section */}
            <div style={{ 
                display: 'grid', 
                gridTemplateColumns: 'repeat(auto-fit, minmax(500px, 1fr))',
                gap: '25px',
                marginBottom: '30px'
            }}>
                <div style={{ 
                    backgroundColor: 'white',
                    padding: '20px',
                    borderRadius: '12px',
                    boxShadow: '0 4px 12px rgba(0,0,0,0.08)'
                }}>
                    <div style={{ 
                        display: 'flex', 
                        justifyContent: 'space-between', 
                        alignItems: 'center',
                        marginBottom: '20px'
                    }}>
                        <h3 style={{ 
                            margin: 0,
                            fontSize: '1.3rem',
                            fontWeight: '600',
                            color: '#2d3748'
                        }}>Usage Trends</h3>
                        {/* <select style={{ 
                            padding: '8px 15px', 
                            borderRadius: '6px',
                            border: '1px solid #e2e8f0',
                            backgroundColor: '#f8fafc',
                            fontSize: '0.95rem',
                            fontWeight: '500'
                        }}>
                            <option>Last 7 Days</option>
                            <option>Last 30 Days</option>
                            <option>Last Quarter</option>
                            <option>Last Year</option>
                        </select> */}
                    </div>
                    <div style={{ height: '350px' }}>
                        <ResponsiveContainer width="100%" height="100%">
                            <LineChart data={usageTrends}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" />
                                <XAxis 
                                    dataKey="name" 
                                    stroke="#718096" 
                                    tick={{ fill: '#4a5568' }} 
                                />
                                <YAxis 
                                    stroke="#718096" 
                                    tick={{ fill: '#4a5568' }} 
                                />
                                <Tooltip 
                                    contentStyle={{ 
                                        backgroundColor: 'rgba(255, 255, 255, 0.9)',
                                        border: '1px solid #e2e8f0',
                                        borderRadius: '8px'
                                    }} 
                                />
                                <Legend />
                                <Line 
                                    type="monotone" 
                                    dataKey="value" 
                                    stroke="#4c6ef5" 
                                    strokeWidth={3}
                                    activeDot={{ r: 8, stroke: '#4c6ef5', strokeWidth: 2, fill: '#fff' }} 
                                    dot={{ r: 4, fill: '#4c6ef5' }}
                                />
                            </LineChart>
                        </ResponsiveContainer>
                    </div>
                </div>

                <div style={{ 
                    backgroundColor: 'white',
                    padding: '20px',
                    borderRadius: '12px',
                    boxShadow: '0 4px 12px rgba(0,0,0,0.08)'
                }}>
                    <div style={{ 
                        display: 'flex', 
                        justifyContent: 'space-between', 
                        alignItems: 'center',
                        marginBottom: '20px'
                    }}>
                        <h3 style={{ 
                            margin: 0,
                            fontSize: '1.3rem',
                            fontWeight: '600',
                            color: '#2d3748'
                        }}>Revenue Heatmap</h3>
                        {/* <select style={{ 
                            padding: '8px 15px', 
                            borderRadius: '6px',
                            border: '1px solid #e2e8f0',
                            backgroundColor: '#f8fafc',
                            fontSize: '0.95rem',
                            fontWeight: '500'
                        }}>
                             <option>Last 7 Days</option>
                            <option>Last 30 Days</option>
                            <option>Last Quarter</option>
                            <option>Last Year</option> 
                        </select> */}
                    </div>
                    <div style={{ height: '350px' }}>
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart data={heatmapData}>
                                <CartesianGrid strokeDasharray="3 3" stroke="#e2e8f0" />
                                <XAxis 
                                    dataKey="name" 
                                    stroke="#718096" 
                                    tick={{ fill: '#4a5568' }} 
                                />
                                <YAxis 
                                    stroke="#718096" 
                                    tick={{ fill: '#4a5568' }} 
                                />
                                <Tooltip 
                                    contentStyle={{ 
                                        backgroundColor: 'rgba(255, 255, 255, 0.9)',
                                        border: '1px solid #e2e8f0',
                                        borderRadius: '8px'
                                    }} 
                                />
                                <Legend />
                                <Bar dataKey="value" name="Revenue">
                                    {heatmapData.map((entry, index) => (
                                        <Cell 
                                            key={`cell-${index}`} 
                                            fill={`rgba(76, 110, 245, ${0.3 + (index * 0.12)})`} 
                                            stroke="#4c6ef5"
                                            strokeWidth={1}
                                        />
                                    ))}
                                </Bar>
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default Dashboard;
