

import React, { useState } from 'react';
import SearchPage from '../../common/SearchPage';
import List from '../../common/List';
import AccountView from './AccountView';
import AccountUpdate from './AccountUpdate';
import AccountDB from './AccountDB';
import { usePermission } from '../../../state/usePermission';
import useCommonEnums from '../../common/CommonData';

const ACCOUNTS_HEADER_TO_KEY_MAP = {
  "Account Title": { 
    key: "title", 
    clickable: true,
    label: "Account Title",
    sortable: true,
    formatter: (value) => value || '-'
  },
  "Name": { 
    key: "account_name", 
    label: "Name",
    sortable: false,
    formatter: (value) => value || '-'
  },
  "Email": { 
    key: "email", 
    label: "Email",
    sortable: false,
    formatter: (value) => value || '-'
  },
  "Type": { 
    key: "subscription_type", 
    label: "Type",
    sortable: false,
    formatter: (value) => value || '-'
  },
  "Date": { 
    key: "created_date", 
    label: "Date",
    sortable: true,
    formatter: (date) => date ? new Date(date).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    }) : '-'
  },
  "Status": { 
    key: "is_active",
    label: "Status",
    sortable: false,
    formatter: (active) => (
      <span style={{ color: active ? 'green' : 'red' }}>
        {active ? 'Active' : 'Inactive'}
      </span>
    )
  }
};

const AccountsList = ({ initialSearchQuery = '', rowsPerPage = 10, featureId, isGlobalAccess, accountId, referredAccountId }) => {
  // Build the fetchDataUrl with referred_account_id
  let fetchDataUrl = '/accounts/';
  if (referredAccountId) {
    fetchDataUrl += `?referred_account_id=${referredAccountId}`;
  } else if (!isGlobalAccess && accountId) {
    fetchDataUrl += `?account_id=${accountId}`;
  }
  
  const [selectedModal, setSelectedModal] = useState(null);

  const { canRead, canEdit, canDelete } = usePermission(featureId);

  const actionFormats = [
    ...(canRead ? [{ label: 'view', modal: true, url: '/accounts/view/{id}' }] : []),
    ...(canEdit ? [{ label: 'edit', modal: true, url: '/accounts/update/{id}' }] : []),
    ...(canDelete ? [
      { label: 'delete', modal: true, url: '/accounts/delete/{id}' },
    ] : []),
  ];

  const handleActionClick = (accountId, action) => {
    const modalType =
      action.label === 'edit' ? 'update' :
      ['delete', 'block'].includes(action.label) ? 'delete' :
      'view';

    setSelectedModal({ accountId, modalType });
  };

  const handleCellClick = (header, item) => {
    const mappingValue = ACCOUNTS_HEADER_TO_KEY_MAP[header];
    
    if (!mappingValue) {
      console.warn('No mapping found for header:', header);
      return;
    }

    if (mappingValue.clickable) {
      setSelectedModal({
        accountId: item.account_id,
        modalType: 'view',
        data: item
      });
    }
  };

  const closeModal = () => setSelectedModal(null);

  // ✅ use enums hook
  const { enums, loading, error } = useCommonEnums();

  // Show loading/error states
  if (loading) return <div>Loading filter options...</div>;
  if (error) return <div>Error loading filter options: {error.message}</div>;
  if (!enums) return <div>No filter data available</div>;

  const searchFields = [
    { name: 'q', type: 'text', placeholder: 'Search...', defaultValue: initialSearchQuery },
    {
      name: 'subscription_status',
      type: 'dropdown',
      placeholder: 'Subscription Status',
      options: [
        { label: 'All', value: '' },
        ...(
          enums.subscription_status?.length > 0
            ? enums.subscription_status
            : ["Trial", "Active", "Unpaid", "Expired", "Cancelled", "Suspended", "Paused"]
        ).map(val => ({ label: val, value: val })),
      ],
    },
    { name: 'reg_date_from', type: 'date', placeholder: 'From' },
    { name: 'reg_date_to', type: 'date', placeholder: 'To' },
    {
      name: 'workforce_count_range',
      type: 'dropdown',
      placeholder: 'Employee Count',
      options: [
        { label: 'All', value: '' },
        ...(enums.workforce_range || []).map(val => ({ label: val, value: val })),
      ],
    },
    {
      name: 'fleet_count_range',
      type: 'dropdown',
      placeholder: 'Fleet Count',
      options: [
        { label: 'All', value: '' },
        ...(enums.fleet_count_range || []).map(val => ({ label: val, value: val })),
      ],
    },
    {
      name: 'revenue_range',
      type: 'dropdown',
      placeholder: 'Revenues',
      options: [
        { label: 'All', value: '' },
        ...(enums.revenues || []).map(val => ({ label: val, value: val })),
      ],
    },
  ];

  return (
    <div>
      <SearchPage
        fetchDataUrl={fetchDataUrl}
        ListComponent={List}
        itemKeyMapping={ACCOUNTS_HEADER_TO_KEY_MAP}
        dataKey="accounts"
        identifierKey="account_id"
        initialSearchQuery={initialSearchQuery}
        rowsPerPage={rowsPerPage}
        actionFormats={actionFormats}
        onActionClick={handleActionClick}
        onCellClick={handleCellClick}
        searchFields={searchFields}
      />

      {selectedModal?.modalType === 'view' && (
        <AccountView 
          accountId={selectedModal.accountId} 
          accountData={selectedModal.data}
          onClose={closeModal} 
        />
      )}
      {selectedModal?.modalType === 'update' && (
        <AccountUpdate 
          accountId={selectedModal.accountId} 
          onClose={closeModal} 
          onUpdateSuccess={() => console.log("Account updated")} 
        />
      )}
      {selectedModal?.modalType === 'delete' && (
        <AccountDB 
          accountId={selectedModal.accountId} 
          onClose={closeModal} 
          onDeleteSuccess={() => console.log("Account deleted")} 
        />
      )}
    </div>
  );
};

export default AccountsList;