import React, { useState, useEffect } from 'react';
import axios from 'axios';
import Modal from '../../Layouts/Modal';
import WorkforceList from '../../pages/Workforce/WorkforceList';
import FleetsList from '../../pages/Fleets/FleetsList';
import TicketList from '../../adminpages/Tickets/TicketList';
import SubscriptionList from '../../adminpages/Subscriptions/SubscriptionsList';
import InvoiceList from '../../adminpages/Invoices/InvoiceList'; // Add this import


const TAB_CONFIG = [
  { id: 'dashboard', label: 'Dashboard', icon: 'dashboard' },
  { id: 'activity', label: 'Activity', icon: 'history' },
  { id: 'payments', label: 'Payments', icon: 'payments' },
  { id: 'subscriptions', label: 'Subscriptions', icon: 'credit_card' },
  { id: 'invoices', label: 'Invoices', icon: 'receipt' },
  { id: 'users', label: 'Workforce', icon: 'group' },
  { id: 'vehicles', label: 'Vehicles', icon: 'local_shipping' },
  { id: 'tickets', label: 'Tickets', icon: 'confirmation_number' },
  { id: 'info', label: 'Info', icon: 'info' },
];

// Helper components
const InfoRow = ({ label, value }) => (
  <div style={{ marginBottom: '8px' }}>
    <strong>{label}:</strong> {value || 'N/A'}
  </div>
);

const CapabilityList = ({ items }) => (
  <ul style={{ margin: 0, paddingLeft: '20px', listStyleType: 'none' }}>
    {items?.map((item, index) => (
      <li key={index} style={{ marginBottom: '4px', position: 'relative', paddingLeft: '15px' }}>
        <span style={{
          position: 'absolute',
          left: 0,
          top: '6px',
          width: '6px',
          height: '6px',
          backgroundColor: '#34c759',
          borderRadius: '50%'
        }} />
        {item}
      </li>
    ))}
  </ul>
);

// Status options with colors
const STATUS_OPTIONS = [
  { value: "open", label: "Open", color: "#ff9800" },
  { value: "pending", label: "Pending", color: "#2196f3" },
  { value: "in_progress", label: "In Progress", color: "#673ab7" },
  { value: "resolved", label: "Resolved", color: "#4caf50" },
  { value: "closed", label: "Closed", color: "#9e9e9e" }
];

// New List Components
const ActivityList = ({ items, onItemClick }) => {
  const ACTIVITY_HEADERS = [
    { key: "activity_title", label: "Activity", sortable: true },
    { key: "status", label: "Status", sortable: true },
    { key: "created_date", label: "Date", sortable: true }
  ];

  return (
    <div style={{ overflowX: 'auto' }}>
      <table style={{ width: '100%', borderCollapse: 'collapse' }}>
        <thead>
          <tr style={{ backgroundColor: '#f8f9fa' }}>
            {ACTIVITY_HEADERS.map(header => (
              <th 
                key={header.key}
                style={{ 
                  padding: '12px 15px', 
                  textAlign: 'left', 
                  borderBottom: '1px solid #e0e0e0',
                  cursor: header.sortable ? 'pointer' : 'default'
                }}
              >
                {header.label}
              </th>
            ))}
          </tr>
        </thead>
        <tbody>
          {items.length > 0 ? (
            items.map(item => (
              <tr 
                key={item.id} 
                style={{ borderBottom: '1px solid #eee', cursor: 'pointer' }}
                onClick={() => onItemClick && onItemClick(item)}
              >
                {ACTIVITY_HEADERS.map(header => (
                  <td key={`${item.id}-${header.key}`} style={{ padding: '12px 15px' }}>
                    {header.key === 'status' ? (
                      <span style={{ 
                        backgroundColor: STATUS_OPTIONS.find(s => s.value === item.status)?.color || '#9e9e9e',
                        color: '#fff',
                        padding: '4px 10px',
                        borderRadius: '12px',
                        fontSize: '0.8em',
                        display: 'inline-block'
                      }}>
                        {STATUS_OPTIONS.find(s => s.value === item.status)?.label || item.status}
                      </span>
                    ) : header.key === 'created_at' ? (
                      new Date(item[header.key]).toLocaleString()
                    ) : (
                      item[header.key] || 'N/A'
                    )}
                  </td>
                ))}
              </tr>
            ))
          ) : (
            <tr>
              <td colSpan={ACTIVITY_HEADERS.length} style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
                No items found
              </td>
            </tr>
          )}
        </tbody>
      </table>
    </div>
  );
};

const EmailList = ({ items, onItemClick }) => {
  const EMAIL_HEADERS = [
    { key: "email_title", label: "Subject", sortable: true },
    { key: "email_from", label: "From", sortable: true },
    { key: "email_to", label: "To", sortable: true },
    { key: "status", label: "Status", sortable: true },
    { key: "created_date", label: "Date", sortable: true }
  ];

  return (
    <div style={{ overflowX: 'auto' }}>
      <table style={{ width: '100%', borderCollapse: 'collapse' }}>
        <thead>
          <tr style={{ backgroundColor: '#f8f9fa' }}>
            {EMAIL_HEADERS.map(header => (
              <th 
                key={header.key}
                style={{ 
                  padding: '12px 15px', 
                  textAlign: 'left', 
                  borderBottom: '1px solid #e0e0e0',
                  cursor: header.sortable ? 'pointer' : 'default'
                }}
              >
                {header.label}
              </th>
            ))}
          </tr>
        </thead>
        <tbody>
          {items.length > 0 ? (
            items.map(item => (
              <tr 
                key={item.id} 
                style={{ borderBottom: '1px solid #eee', cursor: 'pointer' }}
                onClick={() => onItemClick && onItemClick(item)}
              >
                {EMAIL_HEADERS.map(header => (
                  <td key={`${item.id}-${header.key}`} style={{ padding: '12px 15px' }}>
                    {header.key === 'status' ? (
                      <span style={{
                        backgroundColor: item.status === 'success' ? '#e8f5e9' : 
                                        item.status === 'failed' ? '#ffebee' : '#fff8e1',
                        color: item.status === 'success' ? '#2e7d32' : 
                               item.status === 'failed' ? '#c62828' : '#f57f17',
                        padding: '2px 8px',
                        borderRadius: '12px',
                        fontSize: '0.8em'
                      }}>
                        {item.status}
                      </span>
                    ) : header.key === 'created_at' ? (
                      new Date(item[header.key]).toLocaleString()
                    ) : (
                      item[header.key] || 'N/A'
                    )}
                  </td>
                ))}
              </tr>
            ))
          ) : (
            <tr>
              <td colSpan={EMAIL_HEADERS.length} style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
                No items found
              </td>
            </tr>
          )}
        </tbody>
      </table>
    </div>
  );
};

const AccountView = ({ accountId, onClose, accountData: initialAccountData }) => {
  const [accountData, setAccountData] = useState(initialAccountData || null);
  const [loading, setLoading] = useState(!initialAccountData);
  const [error, setError] = useState('');
  const [activeTab, setActiveTab] = useState('dashboard');
  const [hoveredTab, setHoveredTab] = useState(null);
  const [tickets, setTickets] = useState([]);
  const [newComment, setNewComment] = useState({});
  const [isPosting, setIsPosting] = useState({});
  const [statusChanges, setStatusChanges] = useState({});
  const [emailContent, setEmailContent] = useState('');
  const [ticketsLoading, setTicketsLoading] = useState(false);
  const [ticketsError, setTicketsError] = useState('');
  const [activities, setActivities] = useState([]);
  const [emails, setEmails] = useState([]);
  const [newActivity, setNewActivity] = useState({ 
    activity_title: '', 
    status: 'open' 
  });
  const [newEmail, setNewEmail] = useState({
    email_title: '',
    email_description: '',
    status: 'success'
  });
  const [communicationStats, setCommunicationStats] = useState({
    emails: 0,
    calls: 0,
    chats: 0
  });
  const [workforceData, setWorkforceData] = useState([]);
  const [workforceLoading, setWorkforceLoading] = useState(false);
  const [workforceError, setWorkforceError] = useState('');
const [dashboardData, setDashboardData] = useState(null);
const [dashboardLoading, setDashboardLoading] = useState(false);
const [dashboardError, setDashboardError] = useState('');




  const Icon = ({ name, color = '#000', style = {} }) => (
    <span
      className="material-symbols-outlined"
      style={{ fontSize: '24px', color, verticalAlign: 'middle', marginRight: '6px', ...style }}
    >
      {name}
    </span>
  );

  useEffect(() => {
    const fetchAccountData = async () => {
      try {
        setLoading(true);
        setError('');
        
        // Only fetch account data if not provided as prop
        if (!initialAccountData) {
          const accountRes = await axios.get(`/accounts/${accountId}`);
          setAccountData(accountRes.data);
        }

        // Fetch activities and emails in parallel with error handling
        const [activitiesRes, emailsRes] = await Promise.all([
          axios.get('/activity/list/', {
            params: { 
              account_id: accountId, 
              limit: 100 
            }
          }).catch(err => {
            console.error('Error fetching activities:', err);
            return { data: { activities: [] } };
          }),
          axios.get('/emails/list/', {
            params: { 
              account_id: accountId, 
              limit: 100 
            }
          }).catch(err => {
            console.error('Error fetching emails:', err);
            return { data: { emails: [] } };
          })
        ]);
        
        setActivities(activitiesRes.data?.activities || []);
        setEmails(emailsRes.data?.emails || []);
        
        setCommunicationStats({
          emails: emailsRes.data?.emails?.length || 0,
          calls: 0,
          chats: 0
        });
        
      } catch (err) {
        console.error('Error in fetchAccountData:', err);
        setError('Error fetching account data.');
      } finally {
        setLoading(false);
      }
    };

    if (accountId) {
      fetchAccountData();
    }
  }, [accountId, initialAccountData]);

  useEffect(() => {
    const fetchTickets = async () => {
      if (activeTab === 'tickets' && accountId) {
        setTicketsLoading(true);
        try {
          const response = await axios.get('ticket/list/', {
            params: {
              account_id: accountId,
              limit: 100
            }
          });
          setTickets(response.data.tickets || []);
        } catch (err) {
          setTicketsError('Failed to load tickets.');
        } finally {
          setTicketsLoading(false);
        }
      }
    };

    fetchTickets();
  }, [activeTab, accountId]);

   // Add this useEffect for fetching dashboard data
useEffect(() => {
  const fetchDashboardData = async () => {
    if (activeTab === 'dashboard' && accountId) {
      setDashboardLoading(true);
      try {
        const response = await axios.get(`accounts/${accountId}`);
        setDashboardData(response.data);
      } catch (err) {
        setDashboardError('Failed to load dashboard data.');
        console.error('Error fetching dashboard data:', err);
      } finally {
        setDashboardLoading(false);
      }
    }
  };

  fetchDashboardData();
}, [activeTab, accountId]);










  useEffect(() => {
    const fetchWorkforceData = async () => {
      if (activeTab === 'users' && accountId) {
        setWorkforceLoading(true);
        try {
          const response = await axios.get('/workforce/list/', {
            params: {
              account_id: accountId,
              limit: 100
            }
          });
          setWorkforceData(response.data.workforce || []);
        } catch (err) {
          setWorkforceError('Failed to load workforce data.');
          console.error('Error fetching workforce:', err);
        } finally {
          setWorkforceLoading(false);
        }
      }
    };

    fetchWorkforceData();
  }, [activeTab, accountId]);

  const handlePostComment = (ticketId) => {
    if (!newComment[ticketId]?.trim()) return;
    
    setIsPosting({ ...isPosting, [ticketId]: true });
    
    setTimeout(() => {
      const updatedTickets = tickets.map(ticket => {
        if (ticket.id === ticketId) {
          const newCommentObj = {
            id: ticket.comments.length + 1,
            author: "Current User",
            text: newComment[ticketId],
            timestamp: new Date().toISOString(),
            status_change: statusChanges[ticketId] || null
          };
          
          return {
            ...ticket,
            status: statusChanges[ticketId] || ticket.status,
            comments: [...ticket.comments, newCommentObj]
          };
        }
        return ticket;
      });
      
      setTickets(updatedTickets);
      setNewComment({ ...newComment, [ticketId]: '' });
      setStatusChanges({ ...statusChanges, [ticketId]: null });
      setIsPosting({ ...isPosting, [ticketId]: false });
    }, 500);
  };

  const handleCreateActivity = async () => {
    if (!newActivity.activity_title.trim()) return;
    
    try {
      const user = JSON.parse(localStorage.getItem('user'));
      
      const response = await axios.post('/activity/', {
        account_id: accountId,
        activity_title: newActivity.activity_title,
        status: newActivity.status,
        user_id: user?.id || 'system',
        to_id: user?.id || 'system'
      });
      
      setActivities([response.data, ...activities]);
      setNewActivity({ activity_title: '', status: 'open' });
      setError('');
    } catch (err) {
      setError('Failed to create activity');
    }
  };

  const handleSendEmail = async () => {
    if (!newEmail.email_title.trim() || !newEmail.email_description.trim()) return;
    
    try {
      const user = JSON.parse(localStorage.getItem('user'));
      
      const response = await axios.post('/emails/', {
        account_id: accountId,
        status: newEmail.status,
        email_from: user?.email || 'support@example.com',
        email_to: accountData?.email || '',
        email_title: newEmail.email_title,
        email_description: newEmail.email_description,
        user_id: user?.id || 'system'
      });
      
      setEmails([response.data, ...emails]);
      setCommunicationStats(prev => ({
        ...prev,
        emails: prev.emails + 1
      }));
      setNewEmail({ 
        email_title: '', 
        email_description: '', 
        status: 'success' 
      });
      setError('');
    } catch (err) {
      setError('Failed to send email');
    }
  };

  const renderTabContent = () => {
    if (loading) {
      return (
        <div style={{ padding: '20px', textAlign: 'center' }}>
          Loading account data...
        </div>
      );
    }

    if (error) {
      return (
        <div style={{ padding: '20px', color: '#f44336' }}>
          {error}
        </div>
      );
    }

    switch (activeTab) {
      case 'activity':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ 
              display: 'grid', 
              gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', 
              gap: '15px',
              marginBottom: '20px',
              padding: '15px',
              backgroundColor: '#f8f9fa',
              borderRadius: '8px'
            }}>
              <div>
                <h5 style={{ marginBottom: '5px', fontSize: '0.9rem' }}>Total Workforce</h5>
                <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                     {accountData?.workforce_count ?? '0'}
                </div>
              </div>
              <div>
                <h5 style={{ marginBottom: '5px', fontSize: '0.9rem' }}>Total Customers</h5>
                <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                    {accountData?.customer_count ?? '0'}
                </div>
              </div>
              <div>
                <h5 style={{ marginBottom: '5px', fontSize: '0.9rem' }}>Total Fleet</h5>
                <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                    {accountData?.fleet_count ?? '0'}
                </div>
              </div>
              <div>
                <h5 style={{ marginBottom: '5px', fontSize: '0.9rem' }}>Active Workforce</h5>
                <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                    {accountData?.active_workforce ?? '0'}
                </div>
              </div>
              <div>
                <h5 style={{ marginBottom: '5px', fontSize: '0.9rem' }}>Inactive Workforce</h5>
                <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                    {accountData?.inactive_workforce ?? '0'}
                </div>
              </div>
            </div>
            
            <div style={{ 
              marginBottom: '20px', 
              padding: '15px', 
              border: '1px solid #e0e0e0', 
              borderRadius: '8px'
            }}>
              <h5>Create New Activity</h5>
              <div style={{ display: 'flex', gap: '10px', marginBottom: '10px' }}>
                <input
                  type="text"
                  value={newActivity.activity_title}
                  onChange={(e) => setNewActivity({...newActivity, activity_title: e.target.value})}
                  placeholder="Comments..."
                  style={{
                    flex: 1,
                    padding: '8px 12px',
                    border: '1px solid #ddd',
                    borderRadius: '4px'
                  }}
                />
                <select
                  value={newActivity.status}
                  onChange={(e) => setNewActivity({...newActivity, status: e.target.value})}
                  style={{
                    padding: '8px 12px',
                    border: '1px solid #ddd',
                    borderRadius: '4px',
                    width: '100px'
                  }}
                >
                  <option value="open">Open</option>
                  <option value="in_progress">In Progress</option>
                  <option value="resolved">Resolved</option>
                  <option value="closed">Closed</option>
                </select>
              </div>
              <button
                onClick={handleCreateActivity}
                style={{
                  padding: '8px 16px',
                  backgroundColor: '#34c759',
                  color: 'white',
                  border: 'none',
                  borderRadius: '4px',
                  cursor: 'pointer'
                }}
              >
                Create Activity
              </button>
            </div>
            
            <div>
              <h5>Recent Activities</h5>
              <ActivityList 
                items={activities} 
                onItemClick={(activity) => console.log('Activity clicked:', activity)}
              />
            </div>
          </div>
        );

      case 'tickets':
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Tickets for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {tickets.length}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <TicketList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId} // Add this line to pass the accountId
        />
      </div>
    </div>
  );
   // In the AccountView component, update the Vehicles tab section:

      case 'vehicles':
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Fleet for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {accountData?.fleet_count || 0}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <FleetsList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId}
          isEmbedded={true}
        />
      </div>
    </div>
  );
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Fleet for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {accountData?.fleet_count || 0}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <FleetsList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId}
        />
      </div>
    </div>
  );
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Fleet for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {accountData?.fleet_count || 0}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <FleetsList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId}
        />
      </div>
    </div>
  );
    case 'users':
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Workforce for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {workforceData.length}
        </span>
      </div>

      {workforceLoading ? (
        <div style={{ padding: '40px', textAlign: 'center' }}>
          Loading workforce...
        </div>
      ) : workforceError ? (
        <div style={{ padding: '20px', color: '#f44336', textAlign: 'center' }}>
          {workforceError}
        </div>
      ) : (
        <div style={{ 
          border: '1px solid #e0e0e0', 
          borderRadius: '8px',
          overflow: 'hidden'
        }}>
          <WorkforceList 
            initialSearchQuery="" 
            rowsPerPage={10}
            accountId={accountId}
            isEmbedded={true}
          />
        </div>
      )}
    </div>
  );

      case 'communication':
        return (
          <div style={{ padding: '20px' }}>
            <div style={{ 
              display: 'grid', 
              gridTemplateColumns: 'repeat(auto-fit, minmax(150px, 1fr))', 
              gap: '15px',
              marginBottom: '20px',
              padding: '15px',
              backgroundColor: '#f8f9fa',
              borderRadius: '8px'
            }}>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '2rem', fontWeight: 'bold'}}>
                  {communicationStats.emails}
                </div>
                <div>Emails</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '2rem', fontWeight: 'bold' }}>
                  {communicationStats.calls}
                </div>
                <div>Calls</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '2rem', fontWeight: 'bold' }}>
                  {communicationStats.chats}
                </div>
                <div>Chats</div>
              </div>
            </div>
            
            <div style={{ 
              marginBottom: '20px',
              border: '1px solid #e0e0e0',
              borderRadius: '8px',
              padding: '15px'
            }}>
              <h5>Compose Email</h5>
              <input
                type="text"
                value={newEmail.email_title}
                onChange={(e) => setNewEmail({...newEmail, email_title: e.target.value})}
                placeholder="Email Title"
                style={{
                  width: '100%',
                  padding: '10px',
                  border: '1px solid #ddd',
                  borderRadius: '4px',
                  marginBottom: '10px'
                }}
              />
              <textarea
                value={newEmail.email_description}
                onChange={(e) => setNewEmail({...newEmail, email_description: e.target.value})}
                placeholder="Email Content..."
                style={{
                  width: '100%',
                  minHeight: '100px',
                  padding: '10px',
                  border: '1px solid #ddd',
                  borderRadius: '4px',
                  marginBottom: '10px',
                  resize: 'vertical'
                }}
              />
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                <button
                  onClick={handleSendEmail}
                  disabled={!newEmail.email_title.trim() || !newEmail.email_description.trim()}
                  style={{
                    padding: '8px 16px',
                    backgroundColor: '#34c759',
                    color: 'white',
                    border: 'none',
                    borderRadius: '4px',
                    cursor: 'pointer'
                  }}
                >
                  Send
                </button>
              </div>
            </div>
            
            <div>
              <h5>Email History</h5>
              <EmailList 
                items={emails} 
                onItemClick={(email) => console.log('Email clicked:', email)}
              />
            </div>
          </div>
        );

      case 'info':
        

        return (
          <div style={{ padding: '20px' }}>
            <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px', marginBottom: '20px' }}>
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Basic Details</h5>
                <InfoRow label="Account Name" value={accountData?.account_name} />
                <InfoRow label="Title" value={accountData?.title} />
                <InfoRow label="Industry" value={accountData?.industry} />
                <InfoRow label="Account Type" value={accountData?.account_type} />
                <InfoRow label="Status" value={accountData?.is_active ? "Active" : "Inactive"} />
              </div>
              
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Contact Details</h5>
                <InfoRow label="Contact Person" value={accountData?.name} />
                <InfoRow label="Email" value={accountData?.email} />
                <InfoRow label="Mobile" value={accountData?.mobile} />
                <InfoRow label="Website" value={accountData?.website} />
              </div>
            </div>

            <div style={{ backgroundColor: '#f8f9fa', padding: '15px', borderRadius: '8px', marginBottom: '20px' }}>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Subscription</h5>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
                <InfoRow label="Subscription Status" value={accountData?.subscription_status} />
                <InfoRow label="Subscription Type" value={accountData?.subscription_type} />
                <InfoRow label="Created Date" value={accountData?.created_date ? new Date(accountData.created_date).toLocaleDateString() : 'N/A'} />
                <InfoRow label="Last Updated" value={accountData?.updated_date ? new Date(accountData.updated_date).toLocaleDateString() : 'N/A'} />
              </div>
            </div>

            <div style={{ marginBottom: '20px' }}>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Capabilities</h5>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px' }}>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Purposes of Use</h6>
                  <CapabilityList items={accountData?.purposes_of_use} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Fleet Capabilities</h6>
                  <CapabilityList items={accountData?.fleet_capabilities} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Workforce Capabilities</h6>
                  <CapabilityList items={accountData?.workforce_capabilities} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Scale</h6>
                  <InfoRow label="Workforce Count" value={accountData?.workforce_count} />
                  <InfoRow label="Fleet Count" value={accountData?.fleet_count} />
                  <InfoRow label="Revenues" value={accountData?.revenues} />
                </div>
              </div>
            </div>

            <div>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Address</h5>
              <div style={{ backgroundColor: '#f8f9fa', padding: '15px', borderRadius: '8px' }}>
                <InfoRow label="Street" value={accountData?.address?.street} />
                <InfoRow label="City" value={accountData?.address?.city} />
                <InfoRow label="State" value={accountData?.address?.state} />
                <InfoRow label="Country" value={accountData?.address?.country} />
                <InfoRow label="Postal Code" value={accountData?.address?.postal_code} />
              </div>
            </div>
          </div>
        );
       
       case 'dashboard':
return (
    <div style={{ padding: '20px' }}>
      {dashboardLoading ? (
        <div style={{ padding: '40px', textAlign: 'center' }}>
          Loading dashboard data...
        </div>
      ) : dashboardError ? (
        <div style={{ padding: '20px', color: '#f44336', textAlign: 'center' }}>
          {dashboardError}
        </div>
      ) : dashboardData ? (
        <>
          {/* Account header - Simplified */}
        <div
  style={{
    marginBottom: "20px",
    padding: "15px",
    backgroundColor: "#f8f9fa",
    borderRadius: "8px",
    display: "flex", // two-column layout
    justifyContent: "space-between",
    alignItems: "flex-start",
  }}
>
  {/* LEFT SIDE: Account Details */}
  <div style={{ flex: 1, marginRight: "20px" }}>
    <h3 style={{ margin: "0 0 8px 0" }}>
      {dashboardData.account_name}
    </h3>
    <p style={{ margin: "4px 0", color: "#666" }}>
      Industry: {dashboardData.industry || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      Account ID: #{dashboardData.accounts_id || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      Email: {dashboardData.email || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      Mobile: {dashboardData.mobile || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      Website: {dashboardData.website || "-"}
    </p>
  </div>

  {/* RIGHT SIDE: Address */}
  <div style={{ flex: 1 }}>
    <h4 style={{ margin: "0 0 8px 0" }}>Address</h4>
    <p style={{ margin: "4px 0", color: "#666" }}>
      {dashboardData.address?.street || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      {dashboardData.address?.city || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      {dashboardData.address?.state || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      {dashboardData.address?.country || "-"}
    </p>
    <p style={{ margin: "4px 0", color: "#666" }}>
      {dashboardData.address?.postal_code || "-"}
    </p>
  </div>
</div>

       
          {/* Main metrics cards */}
          <div style={{ 
            display: 'grid', 
            gridTemplateColumns: 'repeat(auto-fit, minmax(250px, 1fr))', 
            gap: '15px',
            marginBottom: '20px'
          }}>
        {/* Subscription Card */}
<div
  style={{
    padding: "15px",
    backgroundColor: "#f0f4f8",
    borderRadius: "8px",
    textAlign: "center",   // ✅ Center align everything
  }}
>
  <div
    style={{
      display: "flex",
      alignItems: "center",
      justifyContent: "center",  // ✅ Center the icon + title
      marginBottom: "10px",
    }}
  >
    <Icon name="credit_card" style={{ marginRight: "8px" }} />
    <h4 style={{ margin: 0 }}>Subscription</h4>
  </div>

  {/* Subscription Status */}
  <div
    style={{
      fontSize: "1.2rem",
      fontWeight: "bold",
      marginBottom: "5px",
    }}
  >
    {dashboardData.subscription_status}
  </div>

  {/* Subscription Type */}
  <div
    style={{
      fontSize: "0.9rem",
      color: "#666",
      marginBottom: "8px",
    }}
  >
    {dashboardData.subscription_type}
  </div>

  {/* Trial Remaining Days */}
</div>

{/* Workforce Card */}
<div
  style={{
    padding: "15px",
    backgroundColor: "#f0f4f8",
    borderRadius: "8px",
    textAlign: "center", // ✅ center align all content
  }}
>
  <div
    style={{
      display: "flex",
      alignItems: "center",
      justifyContent: "center", // ✅ center icon + title
      marginBottom: "10px",
    }}
  >
    <Icon name="group" style={{ marginRight: "8px" }} />
    <h4 style={{ margin: 0 }}>Workforce</h4>
  </div>

  {/* Workforce Count */}
  <div
    style={{
      fontSize: "1.5rem",
      fontWeight: "bold",
      marginBottom: "5px",
    }}
  >
    {dashboardData.total_workforce || 0}
  </div>

  {/* Workforce Limit */}
  <div
    style={{
      fontSize: "0.9rem",
      color: "#666",
    }}
  >
    of {dashboardData.workforce_count || "N/A"} limit
  </div>
</div>

            
            {/* Fleet Card */}
        <div
  style={{
    padding: "15px",
    backgroundColor: "#f0f4f8",
    borderRadius: "8px",
    textAlign: "center", // ✅ center everything
  }}
>
  {/* Icon + Title Row */}
  <div
    style={{
      display: "flex",
      alignItems: "center",
      justifyContent: "center", // ✅ center align horizontally
      marginBottom: "10px",
    }}
  >
    <Icon name="local_shipping" style={{ marginRight: "8px" }} />
    <h4 style={{ margin: 0 }}>Fleet</h4>
  </div>

  {/* Fleet Count */}
  <div
    style={{
      fontSize: "1.5rem",
      fontWeight: "bold",
      marginBottom: "5px",
    }}
  >
    {dashboardData.total_fleets || 0}
  </div>

  {/* Fleet Limit */}
  <div
    style={{
      fontSize: "0.9rem",
      color: "#666",
    }}
  >
    of {dashboardData.fleet_count || "N/A"} limit
  </div>
</div>

            
            {/* Revenue Card */}
<div
  style={{
    padding: "15px",
    backgroundColor: "#f0f4f8",
    borderRadius: "8px",
    textAlign: "center", // ✅ center everything
  }}
>
  {/* Icon + Title Row */}
  <div
    style={{
      display: "flex",
      alignItems: "center",
      justifyContent: "center", // ✅ center icon + title
      marginBottom: "10px",
    }}
  >
    <Icon name="attach_money" style={{ marginRight: "8px" }} />
    <h4 style={{ margin: 0 }}>Revenues</h4>
  </div>

  {/* Revenue Value */}
  <div
    style={{
      fontSize: "1.2rem",
      fontWeight: "bold",
    }}
  >
    {dashboardData.revenues || "N/A"}
  </div>
</div>

          </div>

          {/* Recent activities - Only show if we have real data */}
          {activities.length > 0 && (
            <div style={{ 
              padding: '15px',
              border: '1px solid #e0e0e0',
              borderRadius: '8px'
            }}>
              <h4 style={{ margin: '0 0 15px 0', display: 'flex', alignItems: 'center' }}>
                <Icon name="history" style={{ marginRight: '8px' }} />
                Recent Activities
              </h4>
              
              {activities.slice(0, 3).map((activity, index) => (
                <div key={index} style={{ 
                  padding: '12px', 
                  borderLeft: '3px solid #34c759',
                  backgroundColor: '#f8f9fa',
                  marginBottom: '10px',
                  borderRadius: '4px'
                }}>
                  <p style={{ margin: '0 0 5px 0' }}>
                    {activity.activity_title}
                  </p>
                  <div style={{ fontSize: '0.8rem', color: '#666' }}>
                    By <strong>{activity.user_id || 'System'}</strong> on {new Date(activity.created_date).toLocaleDateString()}
                  </div>
                </div>
              ))}
              
              {activities.length > 3 && (
                <div style={{ textAlign: 'center', marginTop: '15px' }}>
                  <button 
                    onClick={() => setActiveTab('activity')}
                    style={{
                      padding: '8px 16px',
                      backgroundColor: 'transparent',
                      border: '1px solid #34c759',
                      color: '#34c759',
                      borderRadius: '4px',
                      cursor: 'pointer'
                    }}
                  >
                    View All Activities
                  </button>
                </div>
              )}
            </div>
          )}
        </>
      ) : (
        <div style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
          No dashboard data available.
        </div>
      )}
    </div>
  );
    return (
    <div style={{ padding: '20px' }}>
      {dashboardLoading ? (
        <div style={{ padding: '40px', textAlign: 'center' }}>
          Loading dashboard data...
        </div>
      ) : dashboardError ? (
        <div style={{ padding: '20px', color: '#f44336', textAlign: 'center' }}>
          {dashboardError}
        </div>
      ) : dashboardData ? (
        <>
          {/* Account header */}
          <div style={{ 
            display: 'flex', 
            justifyContent: 'space-between', 
            alignItems: 'center',
            marginBottom: '20px',
            padding: '15px',
            backgroundColor: '#f8f9fa',
            borderRadius: '8px'
          }}>
            <div>
              <h3 style={{ margin: '0 0 5px 0' }}>{dashboardData.account_name}</h3>
              <p style={{ margin: 0, color: '#666' }}>
                {dashboardData.address?.city}, {dashboardData.address?.country} • #{dashboardData.accounts_id}
              </p>
            </div>
            <div style={{ display: 'flex', gap: '15px' }}>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '1.2rem', fontWeight: 'bold' }}>Users</div>
                <div>{dashboardData.total_workforce || 0}</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '1.2rem', fontWeight: 'bold' }}>Vehicles</div>
                <div>{dashboardData.total_fleets || 0}</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '1.2rem', fontWeight: 'bold' }}>Contacts</div>
                <div>0</div>
              </div>
              <div style={{ textAlign: 'center' }}>
                <div style={{ fontSize: '1.2rem', fontWeight: 'bold' }}>Referral</div>
                <div>0</div>
              </div>
            </div>
          </div>

          {/* Subscription status */}
          <div style={{ 
            display: 'grid', 
            gridTemplateColumns: 'repeat(auto-fit, minmax(200px, 1fr))', 
            gap: '15px',
            marginBottom: '20px'
          }}>
            <div style={{ 
              padding: '15px', 
              backgroundColor: '#e8f5e9', 
              borderRadius: '8px',
              textAlign: 'center'
            }}>
              <h4 style={{ margin: '0 0 10px 0' }}>Subscription</h4>
              <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                {dashboardData.subscription_status}
              </div>
              <div style={{ fontSize: '0.9rem', color: '#666', marginTop: '5px' }}>
                {dashboardData.subscription_type}
              </div>
            </div>
            
            <div style={{ 
              padding: '15px', 
              backgroundColor: '#e3f2fd', 
              borderRadius: '8px',
              textAlign: 'center'
            }}>
              <h4 style={{ margin: '0 0 10px 0' }}>Workforce</h4>
              <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                {dashboardData.total_workforce || 0}
              </div>
              <div style={{ fontSize: '0.9rem', color: '#666', marginTop: '5px' }}>
                of {dashboardData.workforce_count || 'N/A'} limit
              </div>
            </div>
            
            <div style={{ 
              padding: '15px', 
              backgroundColor: '#e8f5e9', 
              borderRadius: '8px',
              textAlign: 'center'
            }}>
              <h4 style={{ margin: '0 0 10px 0' }}>Fleet</h4>
              <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                {dashboardData.total_fleets || 0}
              </div>
              <div style={{ fontSize: '0.9rem', color: '#666', marginTop: '5px' }}>
                of {dashboardData.fleet_count || 'N/A'} limit
              </div>
            </div>
            
            <div style={{ 
              padding: '15px', 
              backgroundColor: '#fff3e0', 
              borderRadius: '8px',
              textAlign: 'center'
            }}>
              <h4 style={{ margin: '0 0 10px 0' }}>Revenues</h4>
              <div style={{ fontSize: '1.5rem', fontWeight: 'bold' }}>
                {dashboardData.revenues || 'N/A'}
              </div>
            </div>
          </div>

          {/* Recent activities */}
          <div style={{ 
            marginBottom: '20px',
            padding: '15px',
            border: '1px solid #e0e0e0',
            borderRadius: '8px'
          }}>
            <h4 style={{ margin: '0 0 15px 0' }}>Recent Activities</h4>
            
            <div style={{ 
              padding: '12px', 
              borderLeft: '3px solid #34c759',
              backgroundColor: '#f8f9fa',
              marginBottom: '10px',
              borderRadius: '4px'
            }}>
              <p style={{ margin: '0 0 5px 0' }}>
                I have spoken to the customer and he says I'm interested
              </p>
              <div style={{ fontSize: '0.8rem', color: '#666' }}>
                By <strong>Suresh</strong> on July 25, 2025
              </div>
            </div>
            
            <div style={{ 
              padding: '12px', 
              borderLeft: '3px solid #2196f3',
              backgroundColor: '#f8f9fa',
              marginBottom: '10px',
              borderRadius: '4px'
            }}>
              <p style={{ margin: '0 0 5px 0' }}>
                Sent an email
              </p>
              <div style={{ fontSize: '0.8rem', color: '#666' }}>
                By <strong>Sathish</strong> on July 24, 2025
              </div>
            </div>
            
            <div style={{ textAlign: 'center', marginTop: '15px' }}>
              <button style={{
                padding: '8px 16px',
                backgroundColor: 'transparent',
                border: '1px solid #34c759',
                color: '#34c759',
                borderRadius: '4px',
                cursor: 'pointer'
              }}>
                View All Activities
              </button>
            </div>
          </div>
        </>
      ) : (
        <div style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
          No dashboard data available.
        </div>
      )}
    </div>
  );



  case 'subscriptions':
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Subscriptions for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {accountData?.subscription_count || 0}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <SubscriptionList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId}
          isEmbedded={true}
        />
      </div>
    </div>
  );

case 'invoices':
  return (
    <div style={{ padding: '20px' }}>
      <div style={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center',
        marginBottom: '20px'
      }}>
        <h5>Invoices for {accountData?.account_name}</h5>
        <span style={{ 
          padding: '4px 12px', 
          backgroundColor: '#e8f5e9', 
          borderRadius: '12px',
          fontSize: '0.9em',
          color: '#2e7d32'
        }}>
          Total: {accountData?.invoice_count || 0}
        </span>
      </div>

      <div style={{ 
        border: '1px solid #e0e0e0', 
        borderRadius: '8px',
        overflow: 'hidden'
      }}>
        <InvoiceList 
          initialSearchQuery="" 
          rowsPerPage={10}
          accountId={accountId}
          isEmbedded={true}
        />
      </div>
    </div>
  );

        return (
          <div style={{ padding: '20px' }}>
            <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px', marginBottom: '20px' }}>
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Basic Details</h5>
                <InfoRow label="Account Name" value={accountData?.account_name} />
                <InfoRow label="Title" value={accountData?.title} />
                <InfoRow label="Industry" value={accountData?.industry} />
                <InfoRow label="Account Type" value={accountData?.account_type} />
                <InfoRow label="Status" value={accountData?.is_active ? "Active" : "Inactive"} />
              </div>
              
              <div>
                <h5 style={{ marginBottom: '10px', color: '#555' }}>Contact Details</h5>
                <InfoRow label="Contact Person" value={accountData?.name} />
                <InfoRow label="Email" value={accountData?.email} />
                <InfoRow label="Mobile" value={accountData?.mobile} />
                <InfoRow label="Website" value={accountData?.website} />
              </div>
            </div>

            <div style={{ backgroundColor: '#f8f9fa', padding: '15px', borderRadius: '8px', marginBottom: '20px' }}>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Subscription</h5>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '15px' }}>
                <InfoRow label="Subscription Status" value={accountData?.subscription_status} />
                <InfoRow label="Subscription Type" value={accountData?.subscription_type} />
                <InfoRow label="Created Date" value={accountData?.created_date ? new Date(accountData.created_date).toLocaleDateString() : 'N/A'} />
                <InfoRow label="Last Updated" value={accountData?.updated_date ? new Date(accountData.updated_date).toLocaleDateString() : 'N/A'} />
              </div>
            </div>

            <div style={{ marginBottom: '20px' }}>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Capabilities</h5>
              <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px' }}>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Purposes of Use</h6>
                  <CapabilityList items={accountData?.purposes_of_use} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Fleet Capabilities</h6>
                  <CapabilityList items={accountData?.fleet_capabilities} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Workforce Capabilities</h6>
                  <CapabilityList items={accountData?.workforce_capabilities} />
                </div>
                <div>
                  <h6 style={{ margin: '8px 0', fontSize: '14px' }}>Scale</h6>
                  <InfoRow label="Workforce Count" value={accountData?.workforce_count} />
                  <InfoRow label="Fleet Count" value={accountData?.fleet_count} />
                  <InfoRow label="Revenues" value={accountData?.revenues} />
                </div>
              </div>
            </div>

            <div>
              <h5 style={{ marginBottom: '10px', color: '#555' }}>Address</h5>
              <div style={{ backgroundColor: '#f8f9fa', padding: '15px', borderRadius: '8px' }}>
                <InfoRow label="Street" value={accountData?.address?.street} />
                <InfoRow label="City" value={accountData?.address?.city} />
                <InfoRow label="State" value={accountData?.address?.state} />
                <InfoRow label="Country" value={accountData?.address?.country} />
                <InfoRow label="Postal Code" value={accountData?.address?.postal_code} />
              </div>
            </div>
          </div>
        );
       






      default:
        return (
          <div style={{ padding: '20px', textAlign: 'center', color: '#666' }}>
            Content for {TAB_CONFIG.find(tab => tab.id === activeTab)?.label} tab is coming soon.
          </div>
        );
    }
  };

  return (
    <Modal onClose={onClose}>
      <div style={{
        display: 'flex',
        alignItems: 'center',
        padding: '10px',
        justifyContent: 'space-between',
        borderBottom: '1px solid #eee'
      }}>
        <div style={{ display: 'flex', alignItems: 'center' }}>
          <div style={{
            width: '60px',
            height: '60px',
            borderRadius: '50%',
            backgroundColor: '#e0e0e0',
            color: '#000',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '20px',
            fontWeight: 'bold',
            marginRight: '16px'
          }}>
            {accountData?.account_name?.charAt(0)?.toUpperCase() || 'A'}
          </div>
          <div>
            <div style={{ fontWeight: '600', fontSize: '20px', color: 'black' }}>
              {accountData?.account_name || 'Account'}
            </div>
            <div style={{ fontSize: '14px', color: '#666', marginTop: '4px' }}>
              {accountData?.industry || 'No industry specified'}
            </div>
          </div>
        </div>
      </div>

      <div style={{
        display: 'flex',
        borderBottom: '1px solid #e0e0e0',
        marginBottom: '0',
        padding: '0 10px',
        overflowX: 'auto',
        scrollbarWidth: 'none',
        msOverflowStyle: 'none'
      }}>
        {TAB_CONFIG.map(tab => (
          <button
            key={tab.id}
            style={{
              padding: '12px 16px',
              border: 'none',
              backgroundColor: 'transparent',
              cursor: 'pointer',
              fontSize: '14px',
              fontWeight: activeTab === tab.id ? '600' : '400',
              color: activeTab === tab.id ? '#34c759' : '#666',
              position: 'relative',
              display: 'flex',
              alignItems: 'center',
              gap: '8px',
              whiteSpace: 'nowrap',
              flexShrink: 0
            }}
            onClick={() => setActiveTab(tab.id)}
            onMouseEnter={() => setHoveredTab(tab.id)}
            onMouseLeave={() => setHoveredTab(null)}
          >
            <Icon name={tab.icon} color={activeTab === tab.id ? '#34c759' : '#666'} />
            {tab.label}
            {activeTab === tab.id && (
              <div style={{
                position: 'absolute',
                bottom: '-1px',
                left: '0',
                right: '0',
                height: '2px',
                backgroundColor: '#34c759'
              }} />
            )}
          </button>
        ))}
      </div>

      <div style={{ 
        maxHeight: 'calc(100vh - 200px)', 
        overflowY: 'auto',
        padding: '0'
      }}>
        {renderTabContent()}
      </div>
    </Modal>
  );
};

export default AccountView;