import React, { useState, useEffect } from 'react';
import axios from 'axios';
import AccountForm from './AccountForm';
import Modal from '../../Layouts/Modal';

const AccountUpdate = ({ accountId, onClose, onUpdateSuccess }) => {
  const [accountData, setAccountData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [statusMessage, setStatusMessage] = useState('');
  const [error, setError] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);

  useEffect(() => {
    axios.get(`/accounts/${accountId}`)
      .then(res => {
        const data = res.data;
        console.log("Account data fetched:", data);
        // Ensure account_id is set (if it's missing, use _id)
        if (!data.account_id) {
          data.account_id = data._id;
        }
        setAccountData(data);
        setLoading(false);
      })
      .catch(err => {
        setError('Error fetching account data.');
        setLoading(false);
      });
  }, [accountId]);

  const handleSubmit = (formData) => {
    if (isSubmitting) return; // Prevent duplicate submissions
    setError('');
    setIsSubmitting(true);
    axios.post(`/accounts/${accountId}`, formData)
      .then(res => {
        setStatusMessage('Account updated successfully!');
        if (onUpdateSuccess) onUpdateSuccess(res.data);
        setTimeout(() => onClose(), 1500);
      })
      .catch(err => {
        setError('Error updating account.');
        setIsSubmitting(false);
      });
  };

  return (
    <Modal title="Edit Account" onClose={onClose}>
      {loading ? (
        <div>Loading account data...</div>
      ) : (
        <>
          {statusMessage && <p className="success">{statusMessage}</p>}
          {error && <p className="error">{error}</p>}
          {/* Only render the form if no success message exists */}
          {!statusMessage && (
            <AccountForm
              initialData={accountData}
              onSubmit={handleSubmit}
              isEdit={true}
            />
          )}
        </>
      )}
    </Modal>
  );
};

export default AccountUpdate;
