

import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { withTheme } from '@rjsf/core';
import MaterialUITheme from '@rjsf/material-ui';
import validator from '@rjsf/validator-ajv8';
import axios from 'axios';
import useCommonEnums from '../../common/CommonData';

const Form = withTheme(MaterialUITheme);

const IconCheckboxesWidget = (props) => {
  const { id, options, value = [], onChange } = props;
  const { enumOptions = [] } = options;

  const handleClick = (optionValue) => {
    let newValue = Array.isArray(value) ? [...value] : [];
    if (newValue.includes(optionValue)) {
      newValue = newValue.filter(v => v !== optionValue);
    } else {
      newValue.push(optionValue);
    }
    onChange(newValue);
  };

  return (
    <div id={id} style={{ display: 'flex', flexWrap: 'wrap', gap: '10px' }}>
      {enumOptions.map(option => {
        const isSelected = value.includes(option.value);
        return (
          <div 
            key={option.value}
            onClick={() => handleClick(option.value)}
            style={{
              border: '1px solid #ccc',
              padding: '10px',
              cursor: 'pointer',
              backgroundColor: isSelected ? '#e0e0e0' : '#fff',
              display: 'flex',
              alignItems: 'center',
              gap: '5px'
            }}
          >
            <span className="material-symbols-outlined">chevron_right</span>
            {option.label}
          </div>
        );
      })}
    </div>
  );
};

const AccountForm = ({ initialData = {}, onSubmit, isEdit = false }) => {
  const navigate = useNavigate();
  const { enums, loading, error } = useCommonEnums();
  const userId = localStorage.getItem('user_id') || '';

  // State management
  const [formData, setFormData] = useState({ user_id: userId, ...initialData });
  const [errorMessage, setErrorMessage] = useState('');
  const [successMessage, setSuccessMessage] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitted, setSubmitted] = useState(false);

  // Loading and error states
  if (loading) return <div>Loading form data...</div>;
  if (error) return <div>Error loading form data: {error.message}</div>;
  if (!enums) return <div>No form data available</div>;

  // Use the correct property names from CommonData.js
  // The transformData function returns these properties:
  // industries, purposes_of_use, fleet_capabilities, workforce_capabilities, workforce_range, fleet_count_range, revenues
  const requiredEnums = ['industries', 'purposes_of_use', 'fleet_capabilities', 'workforce_capabilities', 'workforce_range', 'revenues'];
  const missingEnums = requiredEnums.filter(e => !enums[e] || enums[e].length === 0);
  if (missingEnums.length > 0) {
    return <div>Missing required data fields: {missingEnums.join(', ')}</div>;
  }

  // Schema definition - use the correct property names from CommonData.js
  const schema = {
    title: "Account Information",
    type: "object",
    required: ["user_id", "account_name", "title", "industry", "name", "email", "mobile"],
    properties: {
      user_id: { type: "string", title: "User ID" },
      account_name: { type: "string", title: "Organization Name" },
      title: { type: "string", title: "Your Title" },
      industry: { 
        type: "string", 
        title: "Industry",
        enum: enums.industries, // Changed from Industry to industries
        enumNames: enums.industries // Changed from Industry to industries
      },
      workforce_count: {
        type: "string",
        title: "Number of Employees",
        enum: enums.workforce_range, // Changed from WorkforceCountRange to workforce_range
        enumNames: enums.workforce_range, // Changed from WorkforceCountRange to workforce_range
      },
      fleet_count: {
        type: "string",
        title: "Fleet Size",
        enum: enums.workforce_range, // Using the same range for both workforce and fleet
        enumNames: enums.workforce_range,
      },
      revenues: {
        type: "string",
        title: "Revenues",
        enum: enums.revenues, // Changed from RevenueRange to revenues
        enumNames: enums.revenues, // Changed from RevenueRange to revenues
      },
      address: {
        type: "object",
        title: "Address",
        properties: {
          street: { type: "string", title: "Street" },
          city: { type: "string", title: "City" },
          state: { type: "string", title: "State" },
          country: { type: "string", title: "Country" },
          postal_code: { type: "string", title: "Postal Code" }
        }
      },
      purposes_of_use: {
        type: "array",
        title: "Purpose of Use",
        items: { 
          type: "string", 
          enum: enums.purposes_of_use, // Changed from PurposeOfUse to purposes_of_use
          enumNames: enums.purposes_of_use // Changed from PurposeOfUse to purposes_of_use
        },
        uniqueItems: true
      },
      fleet_capabilities: {
        type: "array",
        title: "Fleet Capabilities",
        items: { 
          type: "string", 
          enum: enums.fleet_capabilities, // Changed from FleetCapability to fleet_capabilities
          enumNames: enums.fleet_capabilities // Changed from FleetCapability to fleet_capabilities
        },
        uniqueItems: true
      },
      workforce_capabilities: {
        type: "array",
        title: "Workforce Capabilities",
        items: { 
          type: "string", 
          enum: enums.workforce_capabilities, // Changed from WorkforceCapability to workforce_capabilities
          enumNames: enums.workforce_capabilities // Changed from WorkforceCapability to workforce_capabilities
        },
        uniqueItems: true
      },
      name: { type: "string", title: "Name" },
      email: { type: "string", title: "Email" },
      mobile: { type: "string", title: "Mobile Number" },
      website: { type: "string", title: "Company Domain (Website)" },
      email_domain: { type: "string", title: "Email Domain" },
      is_active: { type: "boolean", title: "Active", default: true },
      subscription_status: {
        type: "string",
        title: "Subscription Status",
        enum: ["Trial", "Active", "Unpaid", "Expired", "Cancelled", "Suspended", "Paused"],
        default: "Trial"
      }
    }
  };

  if (isEdit) {
    schema.properties.account_id = { type: "string", title: "Account ID" };
  }

  const uiSchema = {
    user_id: { "ui:widget": "hidden" },
    purposes_of_use: { "ui:widget": "IconCheckboxesWidget" },
    fleet_capabilities: { "ui:widget": "IconCheckboxesWidget" },
    workforce_capabilities: { "ui:widget": "IconCheckboxesWidget" }
  };

  if (isEdit) {
    uiSchema.account_id = { "ui:widget": "hidden" };
  }

  const handleSubmit = async ({ formData }) => {
    setErrorMessage('');
    if (isSubmitting) return;
    setIsSubmitting(true);
    
    try {
      const url = isEdit ? `/accounts/${formData.account_id}` : '/accounts/';
      const response = await axios.post(url, formData);
      
      if (response.status === 200) {
        setSuccessMessage(isEdit ? 'Account updated successfully.' : 'Account created successfully.');
        localStorage.setItem('account_id', response.data._id);
        setSubmitted(true);
        if (onSubmit) onSubmit(response.data);
      } else {
        setErrorMessage('An error occurred. Please try again.');
      }
    } catch (error) {
      console.error("Error submitting form:", error);
      setErrorMessage(error.response?.data?.message || 'Failed to submit account. Please check your input.');
    } finally {
      setIsSubmitting(false);
    }
  };

  const resetForm = () => {
    setFormData({ user_id: userId });
    setErrorMessage('');
    setSuccessMessage('');
    setSubmitted(false);
  };

  return (
    <div style={{ maxWidth: '600px', padding: '20px' }}>
      {submitted ? (
        <div>
          <h2>{isEdit ? 'Account Updated' : 'Account Created'}</h2>
          {successMessage && <p style={{ color: 'green' }}>{successMessage}</p>}
          <button onClick={resetForm}>Add Another Account</button>
        </div>
      ) : (
        <>
          {errorMessage && <p style={{ color: 'red' }}>{errorMessage}</p>}
          <Form
            schema={schema}
            uiSchema={uiSchema}
            formData={formData}
            onChange={({ formData }) => setFormData(formData)}
            onSubmit={handleSubmit}
            widgets={{ IconCheckboxesWidget }}
            validator={validator}
            liveValidate={false}
            noHtml5Validate={true}
          >
            <div style={{ textAlign: 'center', marginTop: '10px' }}>
              <button type="submit" disabled={isSubmitting}>
                {isSubmitting ? 'Submitting...' : (isEdit ? 'Update Account' : 'Create Account')}
              </button>
            </div>
          </Form>
        </>
      )}
    </div>
  );
};

export default AccountForm;