from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks, Path
from fastapi.responses import JSONResponse
from typing import List, Optional

from app.db import database
from ...models.cx.ticket import (
    Ticket, TicketBase, TicketCreate, TicketUpdate, 
    TicketResponseList, TicketResponse, TicketReply, TicketReplyResponseList,TicketReplyCreate
)
from ...dependencies.auth import get_current_userdetails
from ...services.cx.ticket import (
    create_ticket_service,
    get_tickets_service,
    read_ticket_service,
    update_ticket_service,
    delete_ticket_service,
    create_ticket_reply_service,
    get_ticket_replies_service
)

router = APIRouter()

@router.post("/", response_model=Ticket)
def create_ticket_endpoint(
    ticket: TicketCreate,
    background_tasks: BackgroundTasks,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Create a new ticket.
    """
    return create_ticket_service(ticket, background_tasks, db)

@router.get("/list/", response_model=TicketResponseList)
def list_tickets(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    account_view: Optional[int] = None,
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: dict = Depends(get_current_userdetails)
):
    """
    List tickets by account.
    """
    print("current_usercurrent_usercurrent_user4545454545454545455");
    print(current_user);
    account_id = current_user.get("account_id")
    return get_tickets_service(account_id, skip, limit, q, account_view, db,current_user)

@router.get("/{ticket_id}", response_model=TicketResponse)
def read_ticket(
    ticket_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Get a specific ticket by ID.
    """
    ticket = read_ticket_service(ticket_id, db)
    if not ticket:
        raise HTTPException(status_code=404, detail="Ticket not found")
    return ticket

@router.put("/{ticket_id}", response_model=Ticket)
def update_ticket(
    ticket_id: str,
    ticket_data: TicketUpdate,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Update a ticket by ID.
    """
    return update_ticket_service(ticket_id, ticket_data, db)

@router.delete("/{ticket_id}", response_model=Ticket)
def delete_ticket(
    ticket_id: str = Path(..., description="The ID of the ticket"),
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Delete a ticket by ID.
    """
    return delete_ticket_service(ticket_id, db)

#Ticket Reply
@router.post("/reply", response_model=TicketReply)
def add_ticket_reply(
    payload: TicketReplyCreate,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Add a reply to a specific ticket.
    """
    return create_ticket_reply_service(payload, db)

@router.get("/reply/list/{ticket_id}", response_model=TicketReplyResponseList)
def get_replies_for_ticket(
    ticket_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    """
    Get all replies for a given ticket.
    """
    return get_ticket_replies_service(ticket_id, db)
