from fastapi import APIRouter, HTTPException, Depends, Request
from typing import Optional
from app.db import database
from app.v1.models.saas.partners import Partner, PartnerCreate, PartnerUpdate, Partners
from app.v1.models.saas.usersmodel import User  # Ensure you have a valid User model
from app.v1.dependencies.auth import get_current_userdetails
from app.v1.services.saas import partners as partner_service  # Adjust path as needed

router = APIRouter()

@router.post("/", response_model=Partner)
def create_partner(
    partner: PartnerCreate, 
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    try:
        created_partner = partner_service.create_partner_service(partner, db)
        return created_partner
    except ValueError as ve:
        raise HTTPException(status_code=400, detail=str(ve))
    except Exception as e:
        raise HTTPException(status_code=500, detail="Internal server error")

@router.get("/", response_model=Partners)
def get_partners(
    skip: int = 0, 
    limit: int = 10, 
    q: Optional[str] = None,
    partner_type: Optional[str] = None,
    created_date_from: Optional[str] = None,
    created_date_to: Optional[str] = None,
    status: Optional[str] = None,
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: User = Depends(get_current_userdetails)
):
    print("partners list 3434343434343343434")
    try:
        result = partner_service.get_partners_service(skip, limit, q, partner_type, created_date_from, created_date_to, status, db, current_user)
        return result
    except Exception as e:
        raise HTTPException(status_code=500, detail="Internal server error")

@router.get("/{partner_id}", response_model=Partner)
def get_partner(
    partner_id: str, 
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    try:
        partner = partner_service.get_partner_service(partner_id, db)
        if partner is None:
            raise HTTPException(status_code=404, detail="Partner not found")
        return partner
    except Exception as e:
        raise HTTPException(status_code=400, detail=f"Error retrieving partner: {str(e)}")

@router.post("/{partner_id}", response_model=Partner)
def update_partner(
    partner_id: str,
    partner: PartnerUpdate,
    request: Request,
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: User = Depends(get_current_userdetails)
):
    try:
        updated_partner = partner_service.update_partner_service(partner_id, partner, current_user, db)
        return updated_partner
    except HTTPException as he:
        raise he
    except Exception as e:
        raise HTTPException(status_code=500, detail="Internal server error")

@router.delete("/{partner_id}", response_model=Partner)
def delete_partner(
    partner_id: str, 
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    try:
        deleted_partner = partner_service.delete_partner_service(partner_id, db)
        if deleted_partner is None:
            raise HTTPException(status_code=404, detail="Partner not found")
        return deleted_partner
    except Exception as e:
        raise HTTPException(status_code=400, detail=f"Error deleting partner: {str(e)}")
