from fastapi import APIRouter, Depends
from datetime import datetime
from typing import Optional

from app.db import database
from app.v1.models.platform.iotdevices import (
    IotDevice, IotDeviceCreate, IotDeviceUpdate, IotDeviceList,IotDeviceWithMappingList,NearbyDevicesRequest
)
from app.v1.services.platform.iotdevices import (
    create_iot_device_service, get_iot_device_service,
    update_iot_device_service, delete_iot_device_service,
    list_iot_devices_service,list_iot_devices_with_mapping_service,list_nearby_devices_service
)

router = APIRouter()

@router.post("/", response_model=IotDevice)
async def create_iot_device(device: IotDeviceCreate, db=Depends(database.get_mongo_db)):    
    return await create_iot_device_service(device, db)

@router.get("/with-mapping", response_model=IotDeviceWithMappingList)
def list_iot_devices_with_mapping(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    device_type: Optional[str] = None,
    status: Optional[str] = None,
    date_from: Optional[str] = None,
    date_to: Optional[str] = None,
    db=Depends(database.get_mongo_db)
):
    return list_iot_devices_with_mapping_service(
        skip, limit, q, device_type, status, date_from, date_to, db
    )

@router.post("/nearby", response_model=IotDeviceWithMappingList, tags=["IoT Devices"])
def list_nearby_devices(
    payload: NearbyDevicesRequest,
    db=Depends(database.get_mongo_db)
): 
    return list_nearby_devices_service(
        payload.latitude,
        payload.longitude,
        payload.radius_km,
        db
    )

@router.get("/", response_model=IotDeviceList)
async def list_iot_devices(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    device_type: Optional[str] = None,
    status: Optional[str] = None,
    date_from: Optional[str] = None,
    date_to: Optional[str] = None,
    db=Depends(database.get_mongo_db)
):
    return await list_iot_devices_service(skip, limit, q, device_type, status, date_from, date_to, db)

@router.get("/{account_id}", response_model=IotDevice)
async def get_iot_device(account_id: str, db=Depends(database.get_mongo_db)):
    return await get_iot_device_service(account_id, db)

@router.post("/{account_id}", response_model=IotDevice)
async def post_iot_device(account_id: str, device: IotDeviceCreate, db=Depends(database.get_mongo_db)):
    return await create_iot_device_service(device, db)

@router.put("/{account_id}", response_model=IotDevice)
async def update_iot_device(account_id: str, update: IotDeviceUpdate, db=Depends(database.get_mongo_db)):
    return await update_iot_device_service(account_id, update, db)

@router.delete("/{account_id}", response_model=IotDevice)
async def delete_iot_device(account_id: str, db=Depends(database.get_mongo_db)):
    return await delete_iot_device_service(account_id, db)

# @router.get("/with-mapping", response_model=IotDeviceWithMappingList)
# async def list_iot_devices_with_mapping(
#     skip: int = 0,
#     limit: int = 10,
#     q: Optional[str] = None,
#     device_type: Optional[str] = None,
#     status: Optional[str] = None,
#     date_from: Optional[str] = None,
#     date_to: Optional[str] = None,
#     db=Depends(database.get_mongo_db)
# ):
#     return await list_iot_devices_with_mapping_service(
#         skip, limit, q, device_type, status, date_from, date_to, db
#     )
