from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks
from app.db import database
from typing import Optional
from datetime import datetime

from app.v1.services.platform.customers import (
    create_customer_service, get_customer_service,
    update_customer_service, delete_customer_service, list_customers_service,get_customers_service
)
from app.v1.dependencies.auth import get_current_userdetails
from app.v1.models.platform.customers import CustomerCreate, CustomerUpdate, Customer, CustomersList

from app.v1.models.saas.usersmodel import UserBase, EmailAction
from app.v1.services.saas.users import (
    create_user_service,
    send_email_verification_service,
    read_user_serviceby_email
)

router = APIRouter()

@router.post("/", response_model=Customer)
async def create_customer(
    background_tasks: BackgroundTasks,
    customer: CustomerCreate,
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    # Create customer first
    result = await create_customer_service(customer, db)

    # Construct user login data
    default_password = "Default@123"

    # Create login user
    email_action = {
        "emails": customer.email,
        "action": "signup",
        "account_id": current_user.get("account_id"),
        "role": 104
    }

    email_action_model = EmailAction(**email_action)
    send_email_verification_service(email_action_model, background_tasks, db)

    resVerificationCode = read_user_serviceby_email(customer.email, db)
    verification_code_res = resVerificationCode.get("verificationCode")

    user = {
        "email": customer.email,
        "password": default_password,
        "verificationCode": verification_code_res,
        "name": f"{customer.first_name} {customer.last_name}",
        "mobile": customer.phone,
        "account_id": customer.account_id,
        "date_of_birth": None,
        "google_connect": "",
        "linkedin_connect": "",
        "token": "",
        "picurl": "",
        "is_active": True,
        "is_verified": True,
        "mobile_verified": False,
        "roles": 104
    }

    user_model = UserBase(**user)
    create_user_service(user_model, background_tasks, db)

    return result

@router.get("/{account_id}", response_model=Customer)
async def get_customer(account_id: str, db=Depends(database.get_mongo_db)):
    print("1191919191919191919191991199")
    return await get_customer_service(account_id, db)

@router.put("/{account_id}", response_model=Customer)
async def update_customer(account_id: str, update: CustomerUpdate, db=Depends(database.get_mongo_db)):
    return await update_customer_service(account_id, update, db)

@router.delete("/{account_id}", response_model=Customer)
async def delete_customer(account_id: str, db=Depends(database.get_mongo_db)):
    return await delete_customer_service(account_id, db)

# @router.get("/", response_model=CustomersList)
# async def list_customers(skip: int = 0, limit: int = 10, db=Depends(database.get_mongo_db)):
#     return await list_customers_service(skip, limit, db)

@router.get("/", response_model=CustomersList)
async def get_customers(
    skip: int = 0,
    limit: int = 10,
    q: str = None,
    customer_type: str = None,
    account_id: str = None,
    from_date: Optional[datetime] = None,
    to_date: Optional[datetime] = None,
    sort_by: Optional[str] = None,          
    sort_order: Optional[str] = "asc",  
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    return get_customers_service(
        skip, limit, q, customer_type, account_id, from_date, to_date, sort_by, sort_order, db, current_user
    )

@router.get("/list", response_model=CustomersList)
async def list_customers(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return await list_customers_service(skip, limit, db)

