from fastapi import APIRouter, Depends, HTTPException, Query, Path
from typing import Optional, List, Dict, Any
from app.db import database
from app.v1.dependencies.auth import get_current_userdetails
from app.v1.models.cx.crm import Comment, AccountDetail, AccountStats, Log
from app.v1.services.cx.crm import (
    add_comment_service,
    delete_comment_service,
    get_comments_service,
    get_account_detail_service,
    update_account_detail_service,
    update_sales_pipeline_service,
    get_account_stats_service,
    get_logs_service
)

router = APIRouter()

# --------------------
# Comments Endpoints
# --------------------
@router.post("/comment/", response_model=Comment)
def post_comment(
    account_id: str,
    comment: str,
    db = Depends(database.get_mongo_db),
    current_user = Depends(get_current_userdetails)
) -> Comment:
    team_member_id = current_user.get("id", "unknown")
    try:
        new_comment = add_comment_service(account_id, team_member_id, comment, db)
        return new_comment
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.delete("/comment/{comment_id}", response_model=Comment)
def delete_comment(
    comment_id: str,
    db = Depends(database.get_mongo_db),
    current_user = Depends(get_current_userdetails)
) -> Comment:
    try:
        deleted = delete_comment_service(comment_id, db)
        return deleted
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/comments/{account_id}", response_model=List[Comment])
def get_comments(
    account_id: str,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1),
    db = Depends(database.get_mongo_db)
) -> List[Comment]:
    try:
        comments = get_comments_service(account_id, skip, limit, db)
        return comments
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# --------------------
# Account Detail Endpoints
# --------------------
@router.get("/account_detail/{account_id}", response_model=AccountDetail)
def read_account_detail(
    account_id: str,
    db = Depends(database.get_mongo_db)
) -> AccountDetail:
    detail = get_account_detail_service(account_id, db)
    if not detail:
        raise HTTPException(status_code=404, detail="Account detail not found")
    return detail

@router.put("/account_detail/{account_id}", response_model=AccountDetail)
def update_account_detail(
    account_id: str,
    update_data: Dict[str, Any],
    db = Depends(database.get_mongo_db)
) -> AccountDetail:
    try:
        detail = update_account_detail_service(account_id, update_data, db)
        return detail
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# --------------------
# Sales Pipeline Endpoint
# --------------------
@router.put("/sales_pipeline/{account_id}", response_model=AccountDetail)
def update_sales_pipeline(
    account_id: str,
    pipeline: List[Dict[str, Any]],
    db = Depends(database.get_mongo_db)
) -> AccountDetail:
    try:
        detail = update_sales_pipeline_service(account_id, pipeline, db)
        return detail
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# --------------------
# Account Stats Endpoint
# --------------------
@router.get("/account_stats/{account_id}", response_model=AccountStats)
def get_account_stats(
    account_id: str,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1),
    db = Depends(database.get_mongo_db)
) -> AccountStats:
    try:
        stats = get_account_stats_service(account_id, db, skip, limit)
        return stats
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

# --------------------
# Logs Endpoint
# --------------------
@router.get("/logs/{account_id}", response_model=List[Log])
def get_logs(
    account_id: str,
    skip: int = Query(0, ge=0),
    limit: int = Query(10, ge=1),
    db = Depends(database.get_mongo_db)
) -> List[Log]:
    try:
        logs = get_logs_service(account_id, skip, limit, db)
        return logs
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))
