from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks, Path
from fastapi.responses import JSONResponse
from typing import Optional

from app.db import database
from ...models.saas.communication_email import (
    CommunicationEmail, CommunicationEmailCreate, CommunicationEmailUpdate,
    CommunicationEmailResponseList, CommunicationEmailResponse
)
from ...dependencies.auth import get_current_userdetails
from ...services.saas.communication_email import (
    create_communication_email_service,
    get_communication_emails_service,
    read_communication_email_service,
    update_communication_email_service,
    delete_communication_email_service
)

router = APIRouter()

@router.post("/", response_model=CommunicationEmail)
def create_email(
    payload: CommunicationEmailCreate,
    background_tasks: BackgroundTasks,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return create_communication_email_service(payload, background_tasks, db)

@router.get("/list/", response_model=CommunicationEmailResponseList)
def list_emails(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    account_view: Optional[int] = None,
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: dict = Depends(get_current_userdetails)
):
    account_id = current_user.get("account_id")
    return get_communication_emails_service(account_id, skip, limit, q, account_view, db,current_user)

@router.get("/{email_id}", response_model=CommunicationEmailResponse)
def read_email(
    email_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    email = read_communication_email_service(email_id, db)
    if not email:
        raise HTTPException(status_code=404, detail="Email record not found")
    return email

@router.put("/{email_id}", response_model=CommunicationEmail)
def update_email(
    email_id: str,
    update: CommunicationEmailUpdate,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return update_communication_email_service(email_id, update, db)

@router.delete("/{email_id}", response_model=CommunicationEmail)
def delete_email(
    email_id: str = Path(..., description="The ID of the email record"),
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return delete_communication_email_service(email_id, db)

@router.get("/status-counts")
def get_email_counts(
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: dict = Depends(get_current_userdetails)
):
    return get_email_status_counts(db)
