from fastapi import APIRouter, Depends, HTTPException, BackgroundTasks, Path
from fastapi.responses import JSONResponse
from typing import List, Optional

from app.db import database
from ...models.saas.activity import (
    Activity, ActivityBase, ActivityCreate, ActivityUpdate,
    ActivityResponseList, ActivityResponse
)
from ...dependencies.auth import get_current_userdetails
from ...services.saas.activity import (
    create_activity_service,
    get_activities_service,
    read_activity_service,
    update_activity_service,
    delete_activity_service
)

router = APIRouter()

@router.post("/", response_model=Activity)
def create_activity_endpoint(
    activity: ActivityCreate,
    background_tasks: BackgroundTasks,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return create_activity_service(activity, background_tasks, db)

@router.get("/list/", response_model=ActivityResponseList)
def list_activities(
    skip: int = 0,
    limit: int = 10,
    q: Optional[str] = None,
    db: database.MongoDB = Depends(database.get_mongo_db),
    current_user: dict = Depends(get_current_userdetails)
):
    account_id = current_user.get("account_id")
    return get_activities_service(account_id, skip, limit, q, db)

@router.get("/{activity_id}", response_model=ActivityResponse)
def read_activity(
    activity_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    activity = read_activity_service(activity_id, db)
    if not activity:
        raise HTTPException(status_code=404, detail="Activity not found")
    return activity

@router.put("/{activity_id}", response_model=Activity)
def update_activity(
    activity_id: str,
    update: ActivityUpdate,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return update_activity_service(activity_id, update, db)

@router.delete("/{activity_id}", response_model=Activity)
def delete_activity(
    activity_id: str = Path(..., description="The ID of the activity"),
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return delete_activity_service(activity_id, db)
