from fastapi import APIRouter, HTTPException, Depends
from app.db import database
from app.v1.models.platform.customer_accounts import CustomerAccount
from app.v1.services.platform import customer_accounts as customer_service
from typing import Dict

router = APIRouter()

@router.post("/", response_model=CustomerAccount)
def create_customer_account(
    data: CustomerAccount,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    try:
        return customer_service.create_customer_account_service(data, db)
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))

@router.get("/{account_id}")
def list_customer_accounts(
    account_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return customer_service.get_customer_accounts_service(account_id, db)

@router.get("/detail/{customer_account_id}", response_model=CustomerAccount)
def get_customer_account(
    customer_account_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    account = customer_service.get_customer_account_service(customer_account_id, db)
    if not account:
        raise HTTPException(status_code=404, detail="Customer account not found")
    return account

@router.put("/{customer_account_id}", response_model=CustomerAccount)
def update_customer_account(
    customer_account_id: str,
    data: CustomerAccount,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return customer_service.update_customer_account_service(customer_account_id, data.dict(exclude_unset=True), db)

@router.delete("/{customer_account_id}")
def delete_customer_account(
    customer_account_id: str,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    return customer_service.delete_customer_account_service(customer_account_id, db)
