from pydantic import BaseModel, Field
from bson import ObjectId
from datetime import datetime
from typing import Optional, List


class ActivityBase(BaseModel):
    account_id: str = Field(..., description="Identifier for the account; required.")
    status: str = Field(..., description="Current status of the activity (e.g., open, pending, closed).")
    activity_title: str = Field(..., description="Title or short label for the activity; required.")
    user_id: str = Field(..., description="Identifier for the user who initiated the activity; required.")
    to_id: str = Field(..., description="Identifier for the target entity (user, record, etc.); required.")


class ActivityCreate(ActivityBase):
    pass


class Activity(ActivityBase):
    id: str
    created_date: datetime = Field(default_factory=datetime.utcnow)

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        populate_by_name = True
        from_attributes = True


class ActivityResponse(BaseModel):
    activity_id: str
    account_id: str
    status: str
    activity_title: str
    user_id: str
    to_id: str
    created_date: datetime


class ActivityResponseList(BaseModel):
    total_count: int
    activities: List[ActivityResponse]


class ActivityUpdate(BaseModel):
    status: Optional[str]
    activity_title: Optional[str]
    to_id: Optional[str]
