from enum import Enum
from pydantic import BaseModel, Field, validator, root_validator
from datetime import datetime
from dateutil import parser  # pip install python-dateutil
from typing import Optional, List
from ..common import Address, SubscriptionStatus, Industry, WorkforceCountRange, RevenueRange, PurposeOfUse, FleetCapability, WorkforceCapability, SubscriptionType, AccountType
# Import the common model


class AccountBase(BaseModel):
    account_name: str
    title: str

    # Business Information
    industry: Optional[Industry] = Industry.TRANSPORTATION_MOBILITY
    workforce_count: WorkforceCountRange = WorkforceCountRange.range_0_10
    fleet_count: WorkforceCountRange = WorkforceCountRange.range_0_10
    revenues: Optional[RevenueRange] = RevenueRange.range_0_1M

    # User Information
    user_id: Optional[str] = ""  # Master User Id
    name: Optional[str] = ""
    email: Optional[str] = ""
    mobile: Optional[str] = ""

    # Account details using the common Address model
    website: Optional[str] = ""
    email_domain: Optional[str] = ""
    address: Optional[Address] = None

    # Status
    is_active: Optional[bool] = True
    created_date: Optional[datetime] = Field(default_factory=datetime.utcnow)
    updated_date: Optional[datetime] = Field(default_factory=datetime.utcnow)
    
    # Subscription information
    account_type: AccountType = Field(default=AccountType.ACCOUNT)
    active_subscription_id: Optional[str] = None
    subscription_status: SubscriptionStatus = Field(default=SubscriptionStatus.trial)
    subscription_type: SubscriptionType = Field(default=SubscriptionType.FREE)

    # Capabilities and Purposes
    purposes_of_use: Optional[List[PurposeOfUse]] = []
    fleet_capabilities: Optional[List[FleetCapability]] = []
    workforce_capabilities: Optional[List[WorkforceCapability]] = []
    account_view: Optional[int] = None 
    referred_account_id: Optional[str] = None 


# Define Account as a subclass of AccountBase.
class Account(AccountBase):
    """A complete Account model that can be extended with extra fields if needed."""
    account_id: Optional[str] = None
    accounts_id: Optional[int] = None    
    total_fleets: Optional[int] = None    
    total_workforce: Optional[int] = None    
    pass

class AccountCreate(AccountBase):
    accounts_id : Optional[int] = None
    pass

class AccountUpdate(AccountBase):
    pass

class AccountList(AccountBase):
    account_id: str

class Accounts(BaseModel):
    total_count: int
    accounts: List[AccountList]

class AccountQueryParams(BaseModel):
    skip: int = 0
    limit: int = 10
    account_id: Optional[str] = None  # Filtering by account_id if needed
    q: Optional[str] = None
    account_type: Optional[str] = None
    is_active: Optional[bool] = None
    subscription_status: Optional[str] = None
    reg_date_from: Optional[str] = None
    reg_date_to: Optional[str] = None
    workforce_count_range: Optional[str] = None
    fleet_count_range: Optional[str] = None
    revenue_range: Optional[str] = None
    referred_account_id: Optional[str] = None
