from pydantic import BaseModel, Field, validator
from bson import ObjectId
from datetime import datetime
from typing import Optional, List
from enum import Enum

# class VehicleType(str, Enum):
#     AUTONOMOUS = "autonomous"
#     DRONE = "drone"
#     SEDAN = "sedan"
#     SUV = "suv"
#     TRUCK = "truck"
#     VAN = "van"

class Status(str, Enum):
    ACTIVE = "active"
    INACTIVE = "inactive"
    MAINTENANCE = "maintenance"

class FuelType(str, Enum):
    PETROL = "petrol"
    DIESEL = "diesel"
    ELECTRIC = "electric"
    HYBRID = "hybrid"
    HYDROGEN = "hydrogen"

class VehicleBase(BaseModel):
    manufacturer_name: str = Field(..., min_length=2, max_length=50)
    vehicle_type: str = Field(..., min_length=2, max_length=50)  # Changed from enum to string
    fuel_type: FuelType
    is_autonomous: bool = Field(default=False)
    can_fly: bool = Field(default=False)
    capacity_jobs: int = Field(..., gt=0)
    number_of_wheels: int = Field(..., ge=0)
    max_speed: Optional[int] = Field(None, gt=0)
    engine_capacity: Optional[int] = Field(None, gt=0)
    status: Status = Field(default=Status.ACTIVE)

class VehicleCreate(VehicleBase):
    account_id: str = Field("", min_length=0)
    user_id: str = Field("", min_length=0)
    model_id: int

class Vehicle(VehicleCreate):
    id: str = Field(default=None, alias="_id")
    created_date: datetime = Field(default_factory=datetime.utcnow)
    updated_date: datetime = Field(default_factory=datetime.utcnow)

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        allow_population_by_field_name = True

    @validator('id', pre=True)
    def convert_objectid(cls, v):
        if isinstance(v, ObjectId):
            return str(v)
        return v

class VehicleResponse(BaseModel):
    vehicle_id: str
    account_id: Optional[str] = Field(None, min_length=0)
    user_id: Optional[str] = Field(None, min_length=0)
    manufacturer_name: str
    vehicle_type: str
    fuel_type: str
    is_autonomous: bool
    can_fly: bool
    capacity_jobs: int
    number_of_wheels: int
    max_speed: Optional[int]
    engine_capacity: Optional[int] = Field(None, gt=0)
    status: Optional[str]
    created_date: datetime
    updated_date: datetime
    model_id: int

    @validator('vehicle_id', pre=True)
    def convert_objectid(cls, v):
        if isinstance(v, ObjectId):
            return str(v)
        return v

class VehicleResponseList(BaseModel):
    total_count: int
    vehicles: List[VehicleResponse]

class VehicleUpdate(BaseModel):
    manufacturer_name: Optional[str] = Field(None, min_length=2, max_length=50)
    vehicle_type: Optional[str] = Field(None, min_length=2, max_length=50)  # Changed from enum to string
    fuel_type: Optional[FuelType] = None
    is_autonomous: Optional[bool] = None
    can_fly: Optional[bool] = None
    capacity_jobs: Optional[int] = Field(None, gt=0)
    number_of_wheels: Optional[int] = Field(None, ge=0)
    max_speed: Optional[int] = Field(None, gt=0)
    engine_capacity: Optional[int] = Field(None, gt=0)
    status: Optional[str] = None
