# services.py
from pydantic import BaseModel, Field, validator
from datetime import datetime
from typing import List, Optional, Dict

class Service(BaseModel):
    service_id: int
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    parent_id: Optional[int] = Field(None, description="Parent service identifier, if any")
    service_title: str = Field(..., description="Title of the service")
    service_description: str = Field(..., description="Detailed description of the service")
    capacity_jobs: int = Field(..., description="Maximum number of concurrent jobs")
    has_cost: bool = Field(..., description="Indicates if the service is chargeable")
    image: str = Field(..., description="URL or path for the service image")
    map_icon: str = Field(..., description="URL or path for the map icon")
    status: str = Field(..., description="Status of the service (e.g., active, inactive)")
    is_vehicle_focused: bool = Field(..., description="True if the service is vehicle focused")
    input_variables: List[str] = Field(default_factory=list, description="Input form variables")
    input_addons: List[str] = Field(default_factory=list, description="Additional optional variables")
    checklist: List[str] = Field(default_factory=list, description="Checklist variables")
    skills: List[str] = Field(default_factory=list, description="Skills variables")
    created_date: datetime = Field(default_factory=datetime.utcnow, description="Creation timestamp")
    updated_date: datetime = Field(default_factory=datetime.utcnow, description="Last updated timestamp")

class PricingRule(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    rule_name: str = Field(..., description="Rule identifier")
    pricing_type: str = Field(..., description="Pricing type (e.g., fixed, time, distance)")
    base_cost: float = Field(..., description="Fixed base cost")
    rate: float = Field(..., description="Variable cost per unit")
    unit: str = Field(..., description="Unit for pricing (e.g., minute, km)")
    conditions: List[str] = Field(default_factory=list, description="Conditions or tags (e.g., night, peak)")
    created_date: datetime = Field(default_factory=datetime.utcnow, description="Creation timestamp")
    updated_date: datetime = Field(default_factory=datetime.utcnow, description="Last updated timestamp")

class PricingType(BaseModel):
    account_id: str = Field(..., description="Identifier for the account")
    user_id: str = Field(..., description="Identifier for the user")
    pricing_type: str = Field(..., description="Pricing type value")
    created_datetime: datetime = Field(default_factory=datetime.utcnow, description="Creation timestamp")
    status: str = Field(..., description="Status (e.g., active, inactive, deprecated)")




class WorkforceBase(BaseModel):

    service_title: str
    service_description: str
    skills: List[str] 
    addons: List[str] 
    checklist: List[str] 
    capacity_jobs: int
    has_cost: int
    is_vehicle_focused: int
    image: Optional[str] = None  # Add these fields
    map_icon: Optional[str] = None
    status: Optional[str] = None
    @validator('skills', 'checklist', 'addons', pre=True)
    def parse_comma_separated_strings(cls, v):
        if isinstance(v, str):
            return [item.strip() for item in v.split(',') if item.strip()]
        return v

class WorkforceUpdateBase(BaseModel):

    # Optional fields
    name: Optional[str] = None
    mobile: Optional[str] = None

# Define Workforce as a subclass of WorkforceBase.
class Workforce(WorkforceBase):
    """A complete Workforce model that can be extended with extra fields if needed."""
    account_id: Optional[str] = None

class WorkforceCreate(WorkforceBase):
    user_id: Optional[str] = None
    account_id: Optional[str] = None
    service_id: Optional[str] = None
    s_id: Optional[int] = None
    service_title: Optional[str] = None
    service_description: Optional[str] = None
    capacity_jobs: Optional[int] = None
    has_cost: Optional[bool] = None
    dynamicFields: List[Dict]  # Expect a list of dictionaries
    category_inputs: Optional[List[Dict]] = []
    image: Optional[str] = None
    map_icon: Optional[str] = None
    status: Optional[str] = None
    is_vehicle_focused: Optional[bool] = None
    @validator('skills','addons')
    def validate_skills(cls, v):
        if not v:
            raise ValueError("At least one skill must be selected")
        return v

class WorkforceUpdate(BaseModel):
    account_id: Optional[str] = None
    user_id: Optional[str] = None
    rule_name: Optional[str] = None
    pricing_type: Optional[str] = None
    base_cost: Optional[float] = None
    rate: Optional[float] = None
    unit: Optional[float] = None
    conditions: Optional[str] = None
    last_updated: Optional[datetime] = Field(default_factory=datetime.utcnow)
    image: Optional[str] = None
    map_icon: Optional[str] = None

class WorkforceList(BaseModel):
    user_id: Optional[str] = None
    id: Optional[str] = None
    account_id: Optional[str] = None
    service_id: Optional[str] = None
    s_id: Optional[int] = None
    service_title: Optional[str] = None
    service_description: Optional[str] = None
    capacity_jobs: Optional[int] = None
    has_cost: Optional[bool] = None
    image: Optional[str] = None
    map_icon: Optional[str] = None
    status: Optional[str] = None
    is_vehicle_focused: Optional[bool] = None
    input_variables: Optional[str] = None
    input_addons: Optional[str] = None    
    created_date: Optional[datetime] = None
    last_login: Optional[datetime] = None
    skills: List[str] = []
    checklist: List[str] = []
    addons: List[str] = []  # Add this field if needed

# class WorkforceList(WorkforceBase):
#     workforce_id: str

class Workforces(BaseModel):
    total_count: int
    users: List[WorkforceList]

class CheckList(BaseModel):
    user_id: Optional[str] = None
    account_id: Optional[str] = None
    name: Optional[str] = None

class CheckListSelect(BaseModel):
    users: List[CheckList]

class AddonList(BaseModel):
    user_id: Optional[str] = None
    account_id: Optional[str] = None
    name: Optional[str] = None

class AddonsSelect(BaseModel):
    users: List[AddonList]

class SkillsList(BaseModel):
    user_id: Optional[str] = None
    account_id: Optional[str] = None
    name: Optional[str] = None

class SkillsListSelect(BaseModel):
    users: List[SkillsList]

class ServiceCatList(BaseModel):
    user_id: Optional[str] = None
    account_id: Optional[str] = None
    service_category: Optional[str] = None

class ServiceCategorySelect(BaseModel):
    users: List[ServiceCatList]

class SubServiceSelect(BaseModel):
    users: List[WorkforceList]
    