from pydantic import BaseModel, Field
from bson import ObjectId
from datetime import datetime
from typing import Optional, List


class TicketBase(BaseModel):
    subject: str = Field(..., description="Subject of the ticket; required.")
    questings: str = Field(..., description="Additional ticket-related queries or tags; required.")
    description: str = Field(..., description="Detailed description of the issue or request; required.")
    account_id: str = Field(..., description="Identifier for the account; required.")
    comment: str = Field(..., description="Initial or system-generated comment on the ticket; required.")
    status: str = Field(..., description="Current status of the ticket (e.g., open, pending, closed).")


class TicketCreate(TicketBase):
    pass


class Ticket(TicketBase):
    id: str
    created_date: datetime = Field(default_factory=datetime.utcnow)

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        allow_population_by_field_name = True


class TicketResponse(BaseModel):
    ticket_id: str
    subject: str
    questings: str
    description: str
    account_id: str
    comment: str
    status: str
    created_date: datetime


class TicketResponseList(BaseModel):
    total_count: int
    tickets: List[TicketResponse]


class TicketUpdate(BaseModel):
    subject: Optional[str]
    questings: Optional[str]
    description: Optional[str]
    comment: Optional[str]
    status: Optional[str]

# Ticket Reply
class TicketReplyBase(BaseModel):
    user_id: str = Field(..., description="User sending the reply")
    ticket_id: str = Field(..., description="Related ticket ID")
    account_id: str = Field(..., description="Account ID of the user")
    description: str = Field(..., description="Reply content")
    status: str = Field(..., description="Reply status (e.g., sent, seen)")

class TicketReplyCreate(TicketReplyBase):
    pass

class TicketReply(TicketReplyBase):
    id: str
    created_date: datetime = Field(default_factory=datetime.utcnow)

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        populate_by_name = True
        from_attributes = True


class TicketReplyResponse(BaseModel):
    reply_id: str
    user_id: str
    ticket_id: str
    account_id: str
    description: str
    status: str
    created_date: datetime


class TicketReplyResponseList(BaseModel):
    total_count: int
    replies: List[TicketReplyResponse]
