from fastapi import FastAPI, Request, HTTPException
from fastapi.responses import RedirectResponse
from fastapi.middleware.cors import CORSMiddleware
from dotenv import load_dotenv
from app.middleware import InputSanitizationMiddleware
from fastapi.staticfiles import StaticFiles
from starlette.exceptions import HTTPException as StarletteHTTPException
from app.v1.dependencies import auth  # AuthMiddleware
from app.v1.routers.saas import (
    accounts, subscriptions, support, supportcustomer, users, roles,
    login, dashboard, audit, rbac, appflow, partners, common, invoices, zerodha
)
from app.v1.routers import crud, tradeengine, tradeengineGPT, teGPT
from app.v1.routers.cx import crm
import asyncio
from app.v1.background.movers_refresh import movers_refresh_loop

# Load .env
if not load_dotenv():
    print("Could not load .env file or it is empty.")
    exit(1)

app = FastAPI()


origins = [
    "https://localhost:3004",
    "https://cabf1e352845.ngrok-free.app/",
    "https://dev.movex.ai",
    
]
app.add_middleware(auth.AuthMiddleware)
app.add_middleware(
    CORSMiddleware,
    #allow_origins=["*"], 
    allow_origins=origins, #    
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)
app.add_middleware(InputSanitizationMiddleware)

# Create a separate router for endpoints with CORS set to "*"
open_cors_router = FastAPI()

# Apply CORS middleware to this specific router
open_cors_router.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

app.exception_handler(StarletteHTTPException)


# ✅ No need to define open_cors_router unless you're mounting it explicitly

# ✅ Mount public folder
app.mount("/public", StaticFiles(directory="public"), name="public")

# ✅ Include your routers
app.include_router(accounts.router, prefix="/v1/accounts", tags=["Accounts"])
app.include_router(login.router, prefix="/v1/auth", tags=["Login"])
app.include_router(users.router, prefix="/v1/users", tags=["Users"])
app.include_router(roles.router, prefix="/v1/roles", tags=["Roles"])
app.include_router(rbac.router, prefix="/v1/rbac", tags=["RBAC"])
app.include_router(common.router, prefix="/v1/common", tags=["RBAC"])
app.include_router(appflow.router, prefix="/v1/appflow", tags=["Apps"])
app.include_router(partners.router, prefix="/v1/partners", tags=["Partners"])
app.include_router(audit.router, prefix="/v1/audit", tags=["Audit"])
app.include_router(dashboard.router, prefix="/v1/dashboard", tags=["Dashboard"])
app.include_router(subscriptions.router, prefix="/v1/subscriptions", tags=["Subscriptions"])
app.include_router(invoices.router, prefix="/v1/saasinvoices", tags=["SaaS Invoices"])
app.include_router(support.router, prefix="/v1/support", tags=["Support"])
app.include_router(supportcustomer.router, prefix="/v1/supportcustomer", tags=["Customer Support"])
app.include_router(crm.router, prefix="/v1/cx/crm", tags=["CRM"])
app.include_router(crud.router, prefix="/v1/crud", tags=["CRUD"])
app.include_router(zerodha.router, prefix="/v1/settings/zerodha", tags=["Zerodha Settings"])
app.include_router(tradeengineGPT.router, prefix="/v1/trade", tags=["Zerodha Trade Engine"])
app.include_router(teGPT.router, prefix="/v1/tradeco", tags=["ChatGPT Trading Engine"])


@app.on_event("startup")
async def startup_events() -> None:
    """Start background tasks when the FastAPI app boots.

    Currently this kicks off a periodic ET movers refresh loop, which
    scrapes Economic Times top gainers/losers and keeps the `movers`
    collection up to date during IST market hours.
    """
    # Run ET movers refresh in the background; does not block startup.
    asyncio.create_task(movers_refresh_loop(interval_seconds=900))

# ✅ IP-restricted docs access
allowed_ips = ["127.0.0.1"]

@app.middleware("http")
async def restrict_access(request: Request, call_next):
    client_ip = request.client.host
    if request.url.path.startswith("/docs") or request.url.path.startswith("/redoc"):
        if client_ip not in allowed_ips:
            raise HTTPException(status_code=404, detail="Forbidden")
    return await call_next(request)
