from kiteconnect import KiteConnect, KiteTicker
from typing import Optional, Union
import pandas as pd
import logging

logger = logging.getLogger("ZerodhaClient")
logger.setLevel(logging.INFO)


class ZerodhaClient:
    def __init__(self, api_key: str, api_secret: str, access_token: Optional[str] = None):
        self.api_key = api_key
        self.api_secret = api_secret
        self.access_token = access_token
        self.kite = KiteConnect(api_key=self.api_key)

        if self.access_token:
            self.kite.set_access_token(self.access_token)

    def generate_login_url(self) -> str:
        """Return the Kite login URL"""
        return self.kite.login_url()

    def generate_session(self, request_token: str) -> dict:
        """Exchange request_token for access_token"""
        try:
            session_data = self.kite.generate_session(
                request_token=request_token,
                api_secret=self.api_secret
            )
            self.access_token = session_data["access_token"]
            self.kite.set_access_token(self.access_token)
            logger.info("Zerodha session established")
            return session_data
        except Exception as e:
            logger.exception("Failed to generate Zerodha session")
            raise e

    def refresh_token(self, refresh_token: str) -> dict:
        """(Optional) If using refresh token mechanism"""
        try:
            session_data = self.kite.refresh_token(refresh_token)
            self.access_token = session_data["access_token"]
            self.kite.set_access_token(self.access_token)
            logger.info("Access token refreshed")
            return session_data
        except Exception as e:
            logger.exception("Failed to refresh access token")
            raise e

    def place_order(
        self,
        symbol: str,
        quantity: int,
        transaction_type: str,
        exchange: str = "NSE",
        order_type: str = "MARKET",
        product: str = "CNC",
        validity: str = "DAY",
        price: Optional[float] = None,
        trigger_price: Optional[float] = None,
        disclosed_quantity: Optional[int] = None,
        tag: Optional[str] = None
    ) -> str:
        """Place a market order"""
        try:
            payload = {
                "variety": self.kite.VARIETY_REGULAR,
                "tradingsymbol": symbol,
                "exchange": exchange,
                "transaction_type": transaction_type,
                "quantity": quantity,
                "order_type": order_type,
                "product": product,
                "validity": validity,
                "price": price,
                "trigger_price": trigger_price,
                "disclosed_quantity": disclosed_quantity,
                "tag": tag,
            }
            # Remove None values
            payload = {k: v for k, v in payload.items() if v is not None}

            order_id = self.kite.place_order(**payload)
            logger.info(f"Order placed successfully: {order_id}")
            return order_id
        except Exception as e:
            logger.exception(f"Order placement failed for {symbol}")
            raise e

    def get_profile(self) -> dict:
        try:
            return self.kite.profile()
        except Exception as e:
            logger.exception("Failed to fetch profile")
            raise e

    def get_holdings(self) -> list:
        try:
            return self.kite.holdings()
        except Exception as e:
            logger.exception("Error fetching holdings")
            raise e

    def get_positions(self) -> dict:
        try:
            return self.kite.positions()
        except Exception as e:
            logger.exception("Error fetching positions")
            raise e

    def get_order_history(self, order_id: str) -> list:
        try:
            return self.kite.order_history(order_id=order_id)
        except Exception as e:
            logger.exception("Error fetching order history")
            raise e

    def get_orders(self) -> list:
        try:
            return self.kite.orders()
        except Exception as e:
            logger.exception("Error fetching orders")
            raise e

    def get_quote(self, symbols: list) -> dict:
        """symbols = ['NSE:RELIANCE', 'NSE:TCS']"""
        try:
            return self.kite.quote(symbols)
        except Exception as e:
            logger.exception("Error fetching quotes")
            raise e

    def get_historical_data(
        self,
        instrument_token: Union[str, int],
        interval: str,
        from_date: str,
        to_date: str,
        continuous: bool = False,
        oi: bool = False
    ) -> pd.DataFrame:
        """Fetch historical OHLC data"""
        try:
            data = self.kite.historical_data(
                instrument_token=instrument_token,
                from_date=from_date,
                to_date=to_date,
                interval=interval,
                continuous=continuous,
                oi=oi
            )
            df = pd.DataFrame(data)
            if not df.empty:
                df['date'] = pd.to_datetime(df['date'])
                df.set_index('date', inplace=True)
            return df
        except Exception as e:
            logger.error(f"Error fetching historical data: {e}")
            raise e
