# app/v1/services/tradeGPT.py
from typing import List, Dict, Any, Optional
from datetime import datetime, timedelta
import logging
import os
import json
import re

from bson import ObjectId
from fastapi import HTTPException

from app.db import database
from app.v1.services.zerodha.client import ZerodhaClient

# OpenAI v1+ client support
try:
    from openai import OpenAI
except Exception:
    OpenAI = None

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)

OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "")
OPENAI_MODEL = os.getenv("OPENAI_MODEL", "gpt-4o-mini")
LLM_TIMEOUT_SEC = int(os.getenv("LLM_TIMEOUT_SEC", "20"))

# ---- Helpers: OpenAI client ----
def _get_openai_client():
    if OpenAI is None:
        raise RuntimeError("openai package v1+ not installed. Install 'openai' >= 1.0.0.")
    if not OPENAI_API_KEY:
        raise RuntimeError("OPENAI_API_KEY environment variable is not set.")
    return OpenAI(api_key=OPENAI_API_KEY)

# ---- Helpers: Zerodha delegator (defensive) ----
def _call_zerodha(zerodha, method_name: str, *args, **kwargs):
    """
    Defensive call helper: tries method on wrapper, then on zerodha.kite.
    Raises AttributeError with diagnostics if not found.
    """
    try:
        # try wrapper first
        if hasattr(zerodha, method_name):
            return getattr(zerodha, method_name)(*args, **kwargs)
        # try kite attribute
        kite = getattr(zerodha, "kite", None)
        if kite and hasattr(kite, method_name):
            return getattr(kite, method_name)(*args, **kwargs)
        # try simple alternates
        alt = method_name.replace("_", "")
        if hasattr(zerodha, alt):
            return getattr(zerodha, alt)(*args, **kwargs)
        if kite and hasattr(kite, alt):
            return getattr(kite, alt)(*args, **kwargs)
    except Exception as e:
        # re-raise so caller can log
        raise
    # diagnostics
    sample_attrs = []
    try:
        sample_attrs = [a for a in dir(zerodha) if not a.startswith("_")]
    except Exception:
        sample_attrs = []
    sample = ", ".join(sample_attrs[:40]) if sample_attrs else "<no attrs>"
    raise AttributeError(f"Zerodha client missing method '{method_name}'. Sample attrs: {sample}")

# ---- Interval normalization (very small) ----
_VALID_KITE_INTERVALS = {"minute", "3minute", "5minute", "10minute", "15minute", "30minute", "60minute", "day", "week", "month"}
_INTERVAL_ALIASES = {"1minute":"minute", "1min":"minute", "5min":"5minute", "15min":"15minute", "30min":"30minute", "60min":"60minute", "1hour":"60minute", "hour":"60minute"}

def _normalize_interval_for_kite(interval: str) -> str:
    if not interval:
        raise ValueError("Empty interval")
    iv = str(interval).strip().lower()
    iv = _INTERVAL_ALIASES.get(iv, iv)
    if iv in _VALID_KITE_INTERVALS:
        return iv
    # allow numeric-minute like "3minute"
    m = re.match(r"^(\d+)minute$", iv)
    if m and int(m.group(1)) in (1,3,5,10,15,30,60):
        return "minute" if int(m.group(1))==1 else iv
    raise ValueError(f"Unsupported interval: {interval}")

# ---------------- MOVERS / TOP10 (minimal) ----------------
def refresh_movers_service(db) -> Dict[str, Any]:
    """
    Try to populate db['movers'] using project scrapers if available.
    Minimal: if scrapers unavailable, returns empty lists.
    """
    try:
        from app.v1.services.zerolive.list import TopMoversFetcher  # type: ignore
        fetcher = TopMoversFetcher(db)
        gainers, losers = fetcher.fetch_top_movers()
        def _norm(arr):
            if not arr: return []
            out=[]
            for item in arr:
                if isinstance(item, dict):
                    for k in ("symbol","nse_symbol","tradingsymbol","ticker","name"):
                        if item.get(k):
                            out.append(item.get(k)); break
                elif isinstance(item, str):
                    out.append(item)
            return out
        gainers = _norm(gainers); losers = _norm(losers)
    except Exception:
        logger.info("TopMoversFetcher not available or failed; using empty movers")
        gainers = []
        losers = []

    # Build canonical record with current UTC timestamp.
    record: Dict[str, Any] = {
        "type": "latest",
        "fetched_at": datetime.utcnow(),
        "gainers": gainers,
        "losers": losers,
        "merged": list(dict.fromkeys(gainers + losers)),
    }

    # Keep a single up-to-date view for live usage.
    db["movers"].update_one({"type": "latest"}, {"$set": record}, upsert=True)

    # Also append a snapshot into history for analytics/backtesting.
    history_doc = dict(record)
    history_doc["type"] = "snapshot"
    db["movers_history"].insert_one(history_doc)

    return record

def get_top10_service(db) -> List[str]:
    doc = db["movers"].find_one({"type":"latest"}) or {}
    merged = doc.get("merged", []) if doc else []
    return merged[:10]

# ---------------- INSTRUMENT CACHE (minimal helpers) ----------------
def _load_instruments_cache(db) -> List[Dict[str, Any]]:
    try:
        cached = db["zerodha_instruments"].find_one({"type":"nse_equity"})
        if cached and isinstance(cached.get("instruments"), list):
            return cached["instruments"]
        cursor = db["zerodha_instruments"].find({}, {"_id":0})
        instruments = list(cursor)
        if instruments and isinstance(instruments[0], dict) and "tradingsymbol" in instruments[0]:
            return instruments
    except Exception:
        logger.exception("Error reading instrument cache")
    return []

def _get_instrument_token_from_db(db, symbol: str) -> Optional[int]:
    if not symbol:
        return None
    try:
        doc = db["zerodha_instruments"].find_one({"tradingsymbol": symbol})
        if doc and doc.get("instrument_token"):
            return doc.get("instrument_token")
        cached = db["zerodha_instruments"].find_one({"type":"nse_equity"})
        if cached and isinstance(cached.get("instruments"), list):
            for inst in cached["instruments"]:
                if inst.get("tradingsymbol")==symbol or inst.get("symbol")==symbol:
                    return inst.get("instrument_token")
        # case-insensitive
        doc = db["zerodha_instruments"].find_one({"tradingsymbol":{"$regex":f"^{re.escape(symbol)}$","$options":"i"}})
        if doc and doc.get("instrument_token"):
            return doc.get("instrument_token")
    except Exception:
        logger.exception("DB instrument lookup error for %s", symbol)
    return None

def refresh_instruments_service(db, zerodha: ZerodhaClient, force: bool = False) -> Dict[str, Any]:
    try:
        instruments = _call_zerodha(zerodha, "instruments", "NSE")
        normalized = []
        for inst in (instruments or []):
            if not isinstance(inst, dict): continue
            normalized.append({
                "tradingsymbol": inst.get("tradingsymbol") or inst.get("symbol"),
                "instrument_token": inst.get("instrument_token") or inst.get("token"),
                "name": (inst.get("name") or "").upper(),
                "exchange": inst.get("exchange") or "NSE",
                "raw": inst
            })
        db["zerodha_instruments"].update_one({"type":"nse_equity"},{"$set":{"type":"nse_equity","instruments":normalized,"last_updated":datetime.utcnow()}},upsert=True)
        return {"ok":True,"count":len(normalized)}
    except Exception as e:
        logger.exception("Failed to refresh instruments: %s", e)
        raise HTTPException(status_code=502, detail="Failed to refresh instruments from Zerodha")

# ---------------- SNAPSHOT (candles + quote) ----------------
def fetch_candles_and_quote_service(zerodha: ZerodhaClient, token: int, symbol: str, intervals: List[str] = None, days: int = 1, max_bars: int = 500) -> Dict[str,Any]:
    """
    Fetch raw quote + candles and return them as simple arrays.
    No analysis here — raw data only.
    """
    if intervals is None:
        intervals = ["minute", "5minute", "15minute"]
    snapshot = {"symbol": symbol, "timestamp": datetime.utcnow().isoformat(), "quote": {}, "candles": {}, "meta": {}}

    # quote
    try:
        q_raw = _call_zerodha(zerodha, "quote", [f"NSE:{symbol}"])
        if isinstance(q_raw, dict):
            q = q_raw.get(f"NSE:{symbol}", q_raw.get(symbol, {}))
        else:
            q = q_raw or {}
        snapshot["quote"] = q or {}
    except Exception as e:
        logger.warning("quote fetch failed for %s: %s", symbol, e)
        snapshot["quote"] = {}

    # candles
    for interval in intervals:
        try:
            kite_interval = _normalize_interval_for_kite(interval)
        except Exception as e:
            logger.warning("Skipping unsupported interval '%s' for %s: %s", interval, symbol, e)
            snapshot["candles"][interval] = []
            continue

        try:
            data = _call_zerodha(
                zerodha,
                "historical_data",
                instrument_token=token,
                from_date=(datetime.now() - timedelta(days=days)),
                to_date=datetime.now(),
                interval=kite_interval,
                continuous=False,
                oi=False
            ) or []
        except TypeError:
            # some wrappers expose different signature; try common alternate call shape
            try:
                data = _call_zerodha(zerodha, "historical_data", token, kite_interval, days) or []
            except Exception as e:
                logger.warning("historical_data alternate signature failed for %s %s: %s", symbol, interval, e)
                data = []
        except Exception as e:
            logger.warning("historical_data failed for %s %s: %s", symbol, interval, e)
            data = []

        arr = []
        for r in (data or [])[-max_bars:]:
            t = r.get("date")
            if hasattr(t, "isoformat"):
                t = t.isoformat()
            arr.append({
                "t": t,
                "o": r.get("open"),
                "h": r.get("high"),
                "l": r.get("low"),
                "c": r.get("close"),
                "v": r.get("volume")
            })
        snapshot["candles"][interval] = arr

    return snapshot

def create_snapshot_service(db, zerodha: ZerodhaClient, symbol: str) -> Dict[str,Any]:
    """
    Create & persist snapshot: find token (from DB) -> fetch candles & quote -> save.
    Minimal mapping: does not perform fuzzy name mapping here.
    """
    token = _get_instrument_token_from_db(db, symbol)
    if not token:
        # try refreshing instruments once
        try:
            refresh_instruments_service(db, zerodha, force=True)
            token = _get_instrument_token_from_db(db, symbol)
        except Exception:
            token = None

    if not token:
        logger.error("Instrument token not found for symbol %s", symbol)
        raise HTTPException(status_code=404, detail=f"Instrument token not found for symbol {symbol}")

    snapshot = fetch_candles_and_quote_service(zerodha, token, symbol)
    doc = {"symbol": symbol, "snapshot": snapshot, "created_at": datetime.utcnow()}
    res = db["snapshots"].insert_one(doc)
    doc["_id"] = res.inserted_id
    return doc

def get_snapshot_service(db, snapshot_id: str) -> Optional[Dict[str,Any]]:
    try:
        doc = db["snapshots"].find_one({"_id": ObjectId(snapshot_id)})
        return doc
    except Exception as e:
        logger.warning("get_snapshot_service error: %s", e)
        return None

# ---------------- LLM call & simple JSON parsing ----------------
def _extract_json_from_text(text: str) -> Optional[Dict[str,Any]]:
    """
    Extract first JSON object from text. Returns dict or None.
    """
    if not text:
        return None
    try:
        return json.loads(text)
    except Exception:
        pass
    start=None; depth=0
    for i,ch in enumerate(text):
        if ch=="{":
            if start is None: start=i
            depth+=1
        elif ch=="}":
            depth-=1
            if depth==0 and start is not None:
                candidate = text[start:i+1]
                try:
                    return json.loads(candidate)
                except Exception:
                    start=None
    return None

def _validate_llm_response(resp: Dict[str,Any]) -> Dict[str,Any]:
    """
    Minimal validation: ensure decision exists and is one of BUY/SELL/HOLD.
    Keep other fields as-is.
    """
    if not isinstance(resp, dict):
        return {"decision":"HOLD","confidence":"LOW","rationale":["Invalid LLM response format"]}
    decision = str(resp.get("decision","")).upper()
    if decision not in ("BUY","SELL","HOLD"):
        return {"decision":"HOLD","confidence":"LOW","rationale":["Invalid decision value from LLM"]}
    # normalize confidence
    conf = resp.get("confidence","LOW")
    if isinstance(conf,str):
        conf = conf.upper()
        if conf not in ("HIGH","MEDIUM","LOW"): conf="LOW"
    else:
        conf="LOW"
    resp["confidence"] = conf
    resp["decision"] = decision
    # ensure rationale exists
    if "rationale" not in resp:
        resp["rationale"] = ["No rationale provided by model"]
    return resp

def call_chatgpt_analyze_service(snapshot: Dict[str,Any], question: Optional[str]=None) -> Dict[str,Any]:
    """
    Send snapshot JSON to ChatGPT, expect a single JSON object back (model must return JSON).
    Returns validated dict. No local algorithms applied.
    """
    if not OPENAI_API_KEY:
        logger.warning("OPENAI_API_KEY not set — returning fallback HOLD")
        return {"decision":"HOLD","confidence":"LOW","rationale":["LLM not configured"]}

    system_prompt = (
        "You are an intraday trading assistant. You will be given a JSON payload containing a 'snapshot' key "
        "with raw OHLCV arrays and a 'quote' key. Analyze only the provided data and return EXACTLY one JSON object "
        "with keys: decision (BUY|SELL|HOLD), confidence (HIGH|MEDIUM|LOW), optional entry:{low,high}, optional stop_loss, optional targets[], rationale[] . "
        "Return only JSON — no extra commentary. If uncertain, return decision=HOLD and explain why in rationale."
    )
    user_payload = {"snapshot": snapshot, "question": question or ""}

    try:
        client = _get_openai_client()
        resp = client.chat.completions.create(
            model=OPENAI_MODEL,
            messages=[
                {"role":"system","content": system_prompt},
                {"role":"user","content": json.dumps(user_payload, default=str)}
            ],
            temperature=0.0,
            max_tokens=800,
            timeout=LLM_TIMEOUT_SEC,
        )
        # defensive extraction of text
        text = ""
        try:
            text = resp.choices[0].message["content"]
        except Exception:
            try:
                text = getattr(resp.choices[0].message, "content", "") or str(resp)
            except Exception:
                text = str(resp)

        parsed = _extract_json_from_text(text)
        if not parsed:
            logger.warning("LLM did not return parseable JSON. Raw excerpt: %s", (text or "")[:500])
            return {"decision":"HOLD","confidence":"LOW","rationale":["LLM did not return JSON"]}
        validated = _validate_llm_response(parsed)
        return validated
    except Exception as e:
        logger.exception("LLM call failed: %s", e)
        return {"decision":"HOLD","confidence":"LOW","rationale":[f"LLM error: {str(e)}"]}

# ---------------- CHAT SERVICE (ChatGPT-only) ----------------
def chat_symbol_service(db, zerodha: ZerodhaClient, user: Dict[str,Any], symbol: str, payload: Dict[str,Any]) -> Dict[str,Any]:
    """
    Create (or load) snapshot, send to ChatGPT, persist chat record, return LLM response.
    No local heuristics or fallback algorithms applied.
    """
    snapshot_id = payload.get("snapshot_id")
    question = (payload.get("question") or "").strip()

    snapshot_doc = None
    if snapshot_id:
        snapshot_doc = get_snapshot_service(db, snapshot_id)
    if not snapshot_doc:
        try:
            snapshot_doc = create_snapshot_service(db, zerodha, symbol)
        except HTTPException:
            raise
        except Exception as e:
            logger.exception("snapshot creation failed for %s: %s", symbol, e)
            raise HTTPException(status_code=500, detail="Snapshot creation failed")

    snapshot = snapshot_doc.get("snapshot") or {}
    has_data = bool(snapshot.get("candles") or snapshot.get("quote"))
    if not has_data:
        logger.warning("Snapshot for %s has no data; skipping LLM.", symbol)
        llm_response = {"decision":"HOLD","confidence":"LOW","rationale":["Snapshot empty"]}
    else:
        llm_response = call_chatgpt_analyze_service(snapshot, question)

    if not isinstance(llm_response, dict) or "decision" not in llm_response:
        llm_response = {"decision":"HOLD","confidence":"LOW","rationale":["Invalid LLM response"]}

    chat_doc = {
        "symbol": snapshot_doc.get("symbol", symbol),
        "snapshot_id": str(snapshot_doc.get("_id")),
        "user_id": str(user.get("_id")),
        "question": question,
        "llm_response": llm_response,
        "created_at": datetime.utcnow()
    }
    db["chats"].insert_one(chat_doc)
    return llm_response

# ---------------- LIVE SIGNALS (ChatGPT-only) ----------------
def get_live_signals_service(db, zerodha: ZerodhaClient, mover: str="gainers", limit: int=50, user_id: Optional[str]=None, question: Optional[str]=None) -> Dict[str,Any]:
    """
    For each mover symbol, create snapshot, call ChatGPT, persist the LLM response.
    Returns list of LLM-decisions for frontend.
    """
    results = []
    try:
        movers_doc = db["movers"].find_one({"type":"latest"}) or {}
        candidates = (movers_doc.get("gainers") if mover=="gainers" else movers_doc.get("losers")) or movers_doc.get("merged") or []
        candidates = [c for c in (candidates or []) if isinstance(c,str)]
        if not candidates:
            logger.info("No movers found for mover=%s", mover)
            return {"results": []}

        for raw_symbol in candidates[:limit]:
            try:
                snapshot_doc = create_snapshot_service(db, zerodha, raw_symbol)
                snapshot = snapshot_doc.get("snapshot") or {}
                llm_decision = call_chatgpt_analyze_service(snapshot, question=question)
            except HTTPException as he:
                logger.warning("create_snapshot_service failed for %s: %s", raw_symbol, str(he.detail if hasattr(he,'detail') else he))
                llm_decision = {"decision":"HOLD","confidence":"LOW","rationale":[f"Snapshot failed: {str(he)}"]}
                snapshot_doc = None
            except Exception as e:
                logger.exception("Error processing %s: %s", raw_symbol, e)
                llm_decision = {"decision":"HOLD","confidence":"LOW","rationale":[f"Error: {str(e)}"]}
                snapshot_doc = None

            # persist lightweight stream doc
            stream_doc = {
                "symbol": raw_symbol,
                "user_id": user_id or None,
                "mover_type": mover,
                "snapshot_id": str(snapshot_doc.get("_id")) if snapshot_doc else None,
                "llm_response": llm_decision,
                "created_at": datetime.utcnow()
            }
            try:
                inserted = db["stream"].insert_one(stream_doc)
                stream_id = str(inserted.inserted_id)
            except Exception:
                logger.exception("Failed to persist stream doc for %s", raw_symbol)
                stream_id = None

            # shape minimal row for frontend
            quote = (snapshot_doc.get("snapshot",{}).get("quote",{}) if snapshot_doc else {}) or {}
            ltp = quote.get("last_price", None)
            row = {
                "id": stream_id,
                "symbol": raw_symbol,
                "decision": llm_decision.get("decision"),
                "confidence": llm_decision.get("confidence"),
                "ltp": ltp,
                "llm": llm_decision,
                "snapshot_id": str(snapshot_doc.get("_id")) if snapshot_doc else None
            }
            results.append(row)

        return {"results": results}
    except Exception as e:
        logger.exception("get_live_signals_service failed: %s", e)
        return {"results": [], "error": str(e)}

# ---------------- ORDER PLACEMENT (delegated, minimal) ----------------
def place_short_order_service(db, zerodha: ZerodhaClient, user: Dict[str,Any], payload: Dict[str,Any]) -> Dict[str,Any]:
    if not payload.get("user_confirmation"):
        raise HTTPException(status_code=400, detail="user_confirmation required")
    user_id = str(user.get("_id"))
    user_settings = db["user_settings"].find_one({"user_id": user_id}) or {}
    if not user_settings.get("allow_shorts", False):
        raise HTTPException(status_code=403, detail="User not permitted to short-sell")

    symbol = payload.get("symbol")
    qty = int(payload.get("quantity",0) or 0)
    price = payload.get("price")
    order_type = payload.get("order_type","MARKET")
    MAX_NOTIONAL = float(os.getenv("MAX_NOTIONAL_PER_ORDER", "200000"))
    notional = (float(price) if price else 0.0) * qty
    if price and notional > MAX_NOTIONAL:
        raise HTTPException(status_code=400, detail="Order exceeds notional limit")
    token = _get_instrument_token_from_db(db, symbol)
    if not token:
        raise HTTPException(status_code=404, detail="Instrument token not found")
    order_payload = {
        "tradingsymbol": symbol,
        "exchange": "NSE",
        "transaction_type": "SELL",
        "quantity": qty,
        "product": "MIS",
        "order_type": order_type.upper(),
    }
    if order_type.upper()=="LIMIT" and price:
        order_payload["price"] = float(price)
    try:
        placed = _call_zerodha(zerodha, "place_order", order_payload)
    except Exception as e:
        logger.exception("Order placement failed: %s", e)
        raise HTTPException(status_code=502, detail=str(e))
    order_doc = {"user_id": user_id, "symbol": symbol, "order_payload": order_payload, "zerodha_response": placed, "created_at": datetime.utcnow()}
    db["orders"].insert_one(order_doc)
    return {"order": order_doc, "zerodha_response": placed}

# ---------------- KITE POSTBACK ----------------
def kite_postback_service(payload: Dict[str,Any]) -> Dict[str,Any]:
    db = database.get_mongo_db()
    db["zerodha_postbacks"].insert_one({"payload": payload, "received_at": datetime.utcnow()})
    status = payload.get("status")
    if status in ("REJECTED","CANCELLED","COMPLETE","TRIGGER_PENDING"):
        alert = {"type":"order_postback","payload":payload,"created_at":datetime.utcnow()}
        db["alerts"].insert_one(alert)
    return {"ok": True}
