# app/v1/services/teGPT.py
"""
Pure ChatGPT-based Trading Engine Services
All trading analysis powered by OpenAI GPT models
"""
import os
import json
import logging
import asyncio
from datetime import datetime, timedelta
from typing import Dict, Any, List, Optional, Tuple
import uuid
from bson import ObjectId

import pandas as pd
from fastapi import HTTPException

from app.v1.services.zerodha.client import ZerodhaClient
from app.v1.services.zerolive.list import get_top_gainer_symbols, get_top_loser_symbols
from app.v1.services.zerodha.indicators import IndicatorCalculator
from app.v1.services.zerodha.strategy_features import StrategyFeatureCalculator
from app.v1.services.gpt_engine import (
    OPENAI_MODEL,
    REQUEST_TIMEOUT,
    validate_openai_setup,
    get_openai_client,
    prepare_market_data_prompt,
    call_chatgpt_analysis,
)

logger = logging.getLogger(__name__)

# Trading limits (kept local to trading/order logic)
MAX_POSITION_SIZE = float(os.getenv("MAX_POSITION_SIZE", "100000"))
MIN_PRICE_VALIDATION = float(os.getenv("MIN_PRICE_VALIDATION", "0.01"))

# Timeframe configuration for Zerodha historical data
# These are the default intervals we fetch per symbol for GPT analysis.
DEFAULT_TIMEFRAMES: List[str] = [
    "5minute",
    "15minute",
    "30minute",
    "day",
]

# How many most-recent candles we keep per timeframe when building
# the snapshot for GPT. Kept small to control token usage.
MAX_CANDLES_PER_TIMEFRAME = int(os.getenv("TEGPT_MAX_CANDLES_PER_TF", "30"))

# ============ ZERODHA CLIENT ============

def get_zerodha_client_service(db, current_user) -> ZerodhaClient:
    """Get Zerodha client for current user"""
    user_id = str(current_user.get("_id"))
    settings = db["zerodha_settings"].find_one({"user_id": user_id})
    
    if not settings:
        raise HTTPException(status_code=404, detail="Zerodha settings not found")
    
    if not settings.get("access_token"):
        raise HTTPException(status_code=403, detail="Zerodha not authenticated")
    
    return ZerodhaClient(
        api_key=settings["api_key"],
        api_secret=settings["api_secret"],
        access_token=settings["access_token"]
    )

# ============ MARKET DATA ============

def get_market_movers_service(db, mover_type: str, limit: int) -> Dict[str, Any]:
    """Get market movers from Economic Times with debug logging"""
    try:
        logger.info(f"🔍 FETCHING TOP {mover_type.upper()} - Starting Economic Times scrape")
        
        # Fetch fresh data from Economic Times
        if mover_type == "gainers":
            symbols = get_top_gainer_symbols(db)
            logger.info(f"📈 SCRAPED GAINERS: {symbols}")
        elif mover_type == "losers": 
            symbols = get_top_loser_symbols(db)
            logger.info(f"📉 SCRAPED LOSERS: {symbols}")
        else:  # both
            gainers = get_top_gainer_symbols(db)
            losers = get_top_loser_symbols(db)
            symbols = gainers + losers
            logger.info(f"📊 SCRAPED BOTH - Gainers: {gainers}, Losers: {losers}")
        
        logger.info(f"✅ FINAL SYMBOLS FOR ANALYSIS: {symbols[:limit]}")
        
        return {
            "type": mover_type,
            "symbols": symbols[:limit],
            "source": "fallback",
            "timestamp": datetime.utcnow()
        }
        
    except Exception as e:
        logger.exception("Failed to get market movers")
        raise HTTPException(status_code=500, detail=f"Market movers error: {str(e)}")

def refresh_instruments_service(db, zerodha_client: ZerodhaClient) -> Dict[str, Any]:
    """Refresh instrument master data from Zerodha"""
    try:
        instruments = zerodha_client.kite.instruments("NSE")
        
        # Clear existing instruments
        db["zerodha_instruments"].delete_many({})
        
        # Insert new instruments
        if instruments:
            db["zerodha_instruments"].insert_many(instruments)
        
        count = len(instruments) if instruments else 0
        timestamp = datetime.utcnow()
        
        # Update metadata
        db["system_meta"].update_one(
            {"key": "instruments_last_updated"},
            {"$set": {"value": timestamp, "count": count}},
            upsert=True
        )
        
        return {"count": count, "timestamp": timestamp}
        
    except Exception as e:
        logger.exception("Failed to refresh instruments")
        raise HTTPException(status_code=500, detail=f"Instrument refresh failed: {str(e)}")

def get_instrument_token(db, symbol: str) -> Optional[int]:
    """Get instrument token for symbol"""
    try:
        instrument = db["zerodha_instruments"].find_one({
            "tradingsymbol": symbol,
            "exchange": "NSE"
        })
        return instrument.get("instrument_token") if instrument else None
    except Exception:
        return None

def fetch_market_data(zerodha_client: ZerodhaClient, symbol: str, timeframes: List[str]) -> Dict[str, Any]:
    """Fetch comprehensive market data for analysis"""
    data = {
        "symbol": symbol,
        "timestamp": datetime.utcnow().isoformat(),
        "quote": {},
        "candles": {},
        "error": None
    }
    
    try:
        logger.info(f"📊 ZERODHA DATA FETCH STARTED for {symbol}")
        
        # Get current quote
        try:
            logger.info(f"📈 Fetching live quote for {symbol}...")
            quote_data = zerodha_client.get_quote([f"NSE:{symbol}"])
            data["quote"] = quote_data.get(f"NSE:{symbol}", {})
            logger.info(f"✅ QUOTE DATA for {symbol}: {data['quote']}")
        except Exception as e:
            logger.error(f"❌ Quote fetch failed for {symbol}: {e}")
            data["quote"] = {}
        
        # Get historical data for each timeframe
        logger.info(f"🔍 Finding instrument token for {symbol}...")
        instruments = zerodha_client.kite.instruments("NSE")
        logger.info(f"📋 Total NSE instruments loaded: {len(instruments) if instruments else 0}")
        
        token = None
        for instrument in instruments or []:
            if instrument.get("tradingsymbol") == symbol:
                token = instrument.get("instrument_token")
                logger.info(f"🎯 FOUND INSTRUMENT TOKEN for {symbol}: {token}")
                break

        # Expose instrument token to downstream consumers (frontend links, etc.)
        data["instrument_token"] = token
        
        if not token:
            logger.error(f"❌ NO INSTRUMENT TOKEN FOUND for {symbol}")
            return data
        
        logger.info(f"📊 Fetching historical data for {symbol} across timeframes: {timeframes}")
        
        if token:
            for timeframe in timeframes:
                try:
                    # Choose a reasonable lookback window per timeframe so that
                    # we can safely extract the latest N candles (configured
                    # via MAX_CANDLES_PER_TIMEFRAME) without pulling
                    # excessive history.
                    if timeframe.endswith("minute"):
                        # Intraday intervals – a few days is enough
                        lookback_days = 5
                    elif timeframe == "day":
                        # For daily candles, use a longer window so we can
                        # still trim to the latest N bars.
                        lookback_days = 90
                    else:
                        # Fallback for any other interval
                        lookback_days = 30

                    from_date = (datetime.now() - timedelta(days=lookback_days)).strftime("%Y-%m-%d")
                    to_date = datetime.now().strftime("%Y-%m-%d")
                    logger.info(f"📅 Fetching {timeframe} data for {symbol} from {from_date} to {to_date}")

                    candles_df = zerodha_client.get_historical_data(
                        instrument_token=token,
                        from_date=from_date,
                        to_date=to_date,
                        interval=timeframe
                    )

                    # Convert DataFrame to list of dicts
                    candles = candles_df.to_dict('records') if not candles_df.empty else []
                    data["candles"][timeframe] = candles[-MAX_CANDLES_PER_TIMEFRAME:] if candles else []

                    logger.info(
                        f"✅ {timeframe} DATA for {symbol}: {len(candles)} candles fetched; "
                        f"using {len(data['candles'][timeframe])} latest"
                    )
                    if candles:
                        latest = candles[-1]
                        logger.info(f"📊 Latest {timeframe} candle: {latest}")

                except Exception as e:
                    logger.error(f"❌ Candle fetch failed for {symbol} {timeframe}: {e}")
                    data["candles"][timeframe] = []

        # Build compact indicator summary per timeframe for GPT / analytics
        indicators_summary: Dict[str, Any] = {}
        for tf, tf_candles in data["candles"].items():
            if not tf_candles:
                continue
            try:
                df_tf = pd.DataFrame(tf_candles)
                summary = IndicatorCalculator.summarize_dataframe(df_tf)
                if summary:
                    indicators_summary[tf] = summary
            except Exception as e:
                logger.error(f"❌ Indicator summary failed for {symbol} {tf}: {e}")

        if indicators_summary:
            data["indicators"] = indicators_summary

        # Daily pivots and Fibonacci levels (if daily candles are present)
        try:
            day_candles = data["candles"].get("day") or []
            if day_candles:
                df_day = pd.DataFrame(day_candles)
                pivots = IndicatorCalculator.calculate_pivot_points(df_day)
                data["pivots"] = {"day": pivots}

                fib_levels = IndicatorCalculator.calculate_fibonacci_levels(df_day)
                if fib_levels:
                    data["fib"] = {"day": fib_levels}
        except Exception as e:
            logger.error(f"❌ Pivot/Fibonacci calculation failed for {symbol}: {e}")

        # Strategy-style numeric features (per timeframe + multi-timeframe alignment)
        try:
            strategies = StrategyFeatureCalculator.summarize_strategies(
                candles_by_timeframe=data.get("candles", {}),
                fib=(data.get("fib", {}) or {}).get("day"),
            )
            if strategies:
                data["strategies"] = strategies
        except Exception as e:
    	    logger.error(f"❌ Strategy feature calculation failed for {symbol}: {e}")
        
        # Summarise data quality
        total_candles = sum(len(c) for c in data["candles"].values()) if data["candles"] else 0
        logger.info(f"🎉 ZERODHA DATA COMPLETE for {symbol}")
        logger.info(
            f"📋 FINAL DATA SUMMARY: Quote: {'✅' if data['quote'] else '❌'}, "
            f"Candles keys: {list(data['candles'].keys())}, Total candles: {total_candles}"
        )

        # If we effectively have *no* usable market data, mark it as an error
        if not data["quote"] and total_candles == 0:
            msg = "No market data available from Zerodha (quote and candles empty)"
            logger.error(f"❌ {msg} for {symbol} - likely auth/token issue")
            data["error"] = msg

        return data
        
    except Exception as e:
        logger.exception(f"Market data fetch failed for {symbol}")
        data["error"] = str(e)
        return data

# ============ CORE SERVICES ============

def analyze_symbol_service(
    db, 
    zerodha_client: ZerodhaClient, 
    symbol: str, 
    timeframes: List[str],
    question: str,
    context: str,
    user_id: str
) -> Dict[str, Any]:
    """Analyze single symbol - ChatGPT DISABLED for testing Zerodha data"""
    
    try:
        logger.info(f"🔍 STARTING ANALYSIS for {symbol}")
        
        # Fetch market data
        market_data = fetch_market_data(zerodha_client, symbol, timeframes)
        
        logger.info(f"📊 ZERODHA MARKET DATA FETCHED for {symbol}:")
        logger.info(f"   📈 Quote Available: {'✅' if market_data.get('quote') else '❌'}")
        logger.info(f"   📊 Candle Data: {list(market_data.get('candles', {}).keys())}")
        
        if market_data.get("quote"):
            quote = market_data["quote"]
            logger.info(f"   💰 LTP: {quote.get('last_price', 'N/A')}")
            logger.info(f"   📈 Day High: {quote.get('ohlc', {}).get('high', 'N/A')}")
            logger.info(f"   📉 Day Low: {quote.get('ohlc', {}).get('low', 'N/A')}")
            logger.info(f"   🔄 Volume: {quote.get('volume', 'N/A')}")
        
        for tf, candles in market_data.get("candles", {}).items():
            if candles:
                logger.info(f"   📊 {tf.upper()}: {len(candles)} candles, Latest: O:{candles[-1].get('open')}, C:{candles[-1].get('close')}")
        
        if market_data.get("error"):
            raise HTTPException(status_code=400, detail=f"Market data error: {market_data['error']}")
        
        # CHATGPT ENABLED - Prepare and send prompt for analysis
        logger.info(f"🤖 CHATGPT ANALYSIS STARTING for {symbol}")
        
        # Prepare prompt with market data
        prompt = prepare_market_data_prompt(symbol, market_data, question, context)
        logger.info(f"📝 PROMPT PREPARED for {symbol} - Length: {len(prompt)} chars")
        
        # Call ChatGPT for analysis
        logger.info(f"🚀 SENDING TO CHATGPT for {symbol}...")
        analysis = call_chatgpt_analysis(prompt)
        
        # Log ChatGPT response in detail
        logger.info(f"🎯 CHATGPT RESPONSE for {symbol}:")
        logger.info(f"   📊 Decision: {analysis.get('decision', 'N/A')}")
        logger.info(f"   🎯 Confidence: {analysis.get('confidence', 'N/A')}")
        logger.info(f"   💰 Price Target: {analysis.get('price_target', 'N/A')}")
        logger.info(f"   🛡️ Stop Loss: {analysis.get('stop_loss', 'N/A')}")
        logger.info(f"   📈 Entry Price: {analysis.get('entry_price', 'N/A')}")
        logger.info(f"   🔍 Rationale: {analysis.get('rationale', [])}")
        
        # Validate decision format for frontend
        if analysis.get('decision') not in ['BUY', 'SELL', 'HOLD']:
            logger.warning(f"⚠️ Invalid decision '{analysis.get('decision')}' for {symbol}, defaulting to HOLD")
            analysis['decision'] = 'HOLD'
        
        # Ensure numeric fields are properly formatted
        current_price = market_data.get("quote", {}).get("last_price", 0)
        if not analysis.get('price_target'):
            analysis['price_target'] = current_price
        if not analysis.get('stop_loss'):
            analysis['stop_loss'] = current_price * 0.95  # 5% stop loss default
        if not analysis.get('entry_price'):
            analysis['entry_price'] = current_price

        # Attach current/last traded price explicitly for frontend
        analysis['current_price'] = current_price or None

        # Fallback risk/reward ratio if not provided by GPT
        if not analysis.get('risk_reward_ratio'):
            try:
                entry = float(analysis.get('entry_price') or 0)
                sl = float(analysis.get('stop_loss') or 0)
                target = float(analysis.get('price_target') or 0)
                rr = None
                if entry and sl and target and sl != entry:
                    if analysis['decision'] == 'BUY':
                        denom = entry - sl
                        if denom > 0:
                            rr = (target - entry) / denom
                    elif analysis['decision'] == 'SELL':
                        denom = sl - entry
                        if denom > 0:
                            rr = (entry - target) / denom
                if rr is not None:
                    analysis['risk_reward_ratio'] = round(rr, 2)
            except Exception:
                # If any conversion fails, leave risk_reward_ratio as-is
                pass
        
        logger.info(f"✅ CHATGPT ANALYSIS FORMATTED for {symbol} - Decision: {analysis['decision']}")
        
        # Attach compact numeric features (indicators, strategies, pivots, fib)
        features = {
            "indicators": market_data.get("indicators", {}),
            "strategies": market_data.get("strategies", {}),
            "pivots": market_data.get("pivots", {}),
            "fib": market_data.get("fib", {}),
        }
        analysis["features"] = features

        # Attach instrument token for frontend chart links when available
        instrument_token = market_data.get("instrument_token")
        if instrument_token is not None:
            analysis["instrument_token"] = instrument_token

        # Store analysis in database
        analysis_doc = {
            "symbol": symbol,
            "user_id": user_id,
            "analysis": analysis,
            "market_data": market_data,
            "features": features,
            "question": question,
            "context": context,
            "timeframes": timeframes,
            "timestamp": datetime.utcnow()
        }
        
        result = db["analyses"].insert_one(analysis_doc)
        analysis["analysis_id"] = str(result.inserted_id)
        analysis["timestamp"] = analysis_doc["timestamp"].isoformat()
        analysis["symbol"] = symbol  # Add symbol to the returned analysis
        
        logger.info(f"✅ ANALYSIS COMPLETE for {symbol} - Testing Mode")
        return analysis
        
    except HTTPException:
        raise
    except Exception as e:
        logger.exception(f"Symbol analysis failed for {symbol}")
        raise HTTPException(status_code=500, detail=str(e))

def bulk_analyze_service(
    db,
    zerodha_client: ZerodhaClient,
    symbols: List[str],
    analysis_type: str,
    timeframes: List[str],
    max_concurrent: int,
    user_id: str
) -> List[Dict[str, Any]]:
    """Bulk analyze multiple symbols with detailed progress logging"""
    
    logger.info(f"🔄 BULK ANALYSIS STARTED - {len(symbols)} symbols, Type: {analysis_type}")
    logger.info(f"📋 SYMBOLS TO ANALYZE: {symbols}")
    
    if not symbols:
        logger.warning("⚠️ NO SYMBOLS PROVIDED for bulk analysis")
        return []
    
    # Prepare questions based on analysis type.
    # For the top-movers /live flow we now always use the neutral
    # "general" variant so there is a single unified strategy: decide
    # between BUY (intraday long), SELL (intraday short), or HOLD
    # (wait/no trade) based purely on the data quality.
    questions = {
        "short_sell": (
            "Provide comprehensive intraday trading analysis for this TOP GAINER. "
            "Decide between BUY (intraday long), SELL (intraday short), or HOLD (no trade). "
            "Use HOLD only when the setup is clearly low quality (score < 30) or the evidence is very mixed. "
            "When the up-move is strong and healthy, prefer BUY; when there are multiple objective reversal/exhaustion "
            "signals (e.g. RSI>70 at resistance, bearish patterns, volume/price divergence), prefer SELL."
        ),
        "long_buy": (
            "Provide comprehensive intraday trading analysis for this TOP LOSER / beaten-down stock. "
            "Decide between BUY (intraday long), SELL (intraday short), or HOLD (no trade). "
            "Use HOLD only when the setup is clearly low quality (score < 30) or there is no clear edge. "
            "When price is deeply oversold near support with improving momentum/accumulation, prefer BUY; "
            "reserve SELL for cases where the downtrend remains very strong with fresh breakdowns."
        ),
        "general": (
            "Provide comprehensive intraday trading analysis with a clear decision: "
            "BUY (intraday long), SELL (intraday short), or HOLD (no trade). "
            "Score the setup 0-100 and avoid HOLD on high-quality setups (score >= 60) unless there is a very specific risk."
        ),
    }

    question = questions.get(analysis_type, questions["general"])
    logger.info(f"❓ ANALYSIS QUESTION TYPE: {analysis_type}")
    
    results = []
    successful_analyses = 0
    failed_analyses = 0
    
    # Process symbols in batches to avoid overwhelming the API
    batch_size = min(max_concurrent, 5)
    logger.info(f"📦 PROCESSING IN BATCHES - Batch size: {batch_size}")
    
    for i in range(0, len(symbols), batch_size):
        batch_symbols = symbols[i:i + batch_size]
        batch_num = (i // batch_size) + 1
        total_batches = (len(symbols) + batch_size - 1) // batch_size
        
        logger.info(f"📦 BATCH {batch_num}/{total_batches}: {batch_symbols}")
        
        for symbol_idx, symbol in enumerate(batch_symbols, 1):
            try:
                logger.info(f"🎯 ANALYZING {symbol} ({symbol_idx}/{len(batch_symbols)} in batch)")
                
                analysis = analyze_symbol_service(
                    db=db,
                    zerodha_client=zerodha_client,
                    symbol=symbol,
                    timeframes=timeframes,
                    question=question,
                    context="general",  # single unified strategy context
                    user_id=user_id
                )
                
                results.append(analysis)
                successful_analyses += 1
                logger.info(f"✅ COMPLETED {symbol} - Decision: {analysis.get('decision', 'N/A')}")
                
            except Exception as e:
                failed_analyses += 1
                logger.error(f"❌ FAILED {symbol}: {str(e)}")
                # Add error result
                results.append({
                    "symbol": symbol,
                    "decision": "HOLD",
                    "confidence": "LOW",
                    "rationale": [f"Analysis failed: {str(e)}"],
                    "error": str(e),
                    "timestamp": datetime.utcnow().isoformat()
                })
        
        # Small delay between batches
        if i + batch_size < len(symbols):
            import time
            time.sleep(1)
    
    # Final summary
    logger.info(f"🏁 BULK ANALYSIS COMPLETE")
    logger.info(f"   ✅ Successful: {successful_analyses}")
    logger.info(f"   ❌ Failed: {failed_analyses}")
    logger.info(f"   📊 Total Results: {len(results)}")
    
    # Log decision summary
    decisions = {}
    for result in results:
        decision = result.get('decision', 'UNKNOWN')
        decisions[decision] = decisions.get(decision, 0) + 1
    
    logger.info(f"🎯 DECISION BREAKDOWN: {decisions}")
    
    return results

def chat_with_stock_service(
    db,
    zerodha_client: ZerodhaClient,
    symbol: str,
    message: str,
    user_id: str,
    conversation_id: Optional[str] = None,
    include_fresh_data: bool = True
) -> Dict[str, Any]:
    """Interactive chat about a specific symbol"""
    
    try:
        if not conversation_id:
            conversation_id = str(uuid.uuid4())
        
        # Get conversation history
        history = list(db["chats"].find({
            "user_id": user_id,
            "symbol": symbol,
            "conversation_id": conversation_id
        }).sort("created_at", 1))
        
        # Fetch fresh market data if requested
        market_data = {}
        if include_fresh_data:
            market_data = fetch_market_data(zerodha_client, symbol, ["5minute", "15minute"])
        
        # Prepare context prompt
        context_prompt = f"""You are chatting with a trader about {symbol}. 

CONVERSATION CONTEXT:
"""
        
        # Add conversation history
        for chat in history[-5:]:  # Last 5 messages for context
            role = "User" if chat.get("role") == "user" else "Assistant"
            context_prompt += f"{role}: {chat.get('message', '')}\n"
        
        context_prompt += f"\nCURRENT USER MESSAGE: {message}\n"
        
        # Add market data if available
        if market_data and not market_data.get("error"):
            quote = market_data.get("quote", {})
            if quote:
                context_prompt += f"""
CURRENT MARKET DATA for {symbol}:
- Price: ₹{quote.get('last_price', 'N/A')}
- Change: {quote.get('net_change', 'N/A')} ({quote.get('net_change_percentage', 'N/A')}%)
- Volume: {quote.get('volume', 'N/A')}
"""
        
        context_prompt += """
Provide a helpful, conversational response about this stock. Be specific and actionable.
If the user asks about technical indicators or patterns, analyze the available data.
Keep responses concise but informative.
"""
        
        # Get ChatGPT response
        try:
            client = get_openai_client()
            
            response = client.chat.completions.create(
                model=OPENAI_MODEL,
                messages=[
                    {"role": "system", "content": "You are an expert stock trader providing helpful analysis and advice."},
                    {"role": "user", "content": context_prompt}
                ],
                temperature=0.3,
                max_tokens=500,
                timeout=REQUEST_TIMEOUT
            )
            
            assistant_message = response.choices[0].message.content.strip()
            
        except Exception as e:
            logger.exception("ChatGPT call failed in chat")
            assistant_message = f"I'm having trouble analyzing {symbol} right now. Please try again in a moment."
        
        # Store user message
        user_chat_doc = {
            "user_id": user_id,
            "symbol": symbol,
            "conversation_id": conversation_id,
            "role": "user",
            "message": message,
            "created_at": datetime.utcnow()
        }
        db["chats"].insert_one(user_chat_doc)
        
        # Store assistant response
        assistant_chat_doc = {
            "user_id": user_id,
            "symbol": symbol,
            "conversation_id": conversation_id,
            "role": "assistant",
            "message": assistant_message,
            "market_data_included": include_fresh_data,
            "created_at": datetime.utcnow()
        }
        result = db["chats"].insert_one(assistant_chat_doc)
        
        return {
            "conversation_id": conversation_id,
            "message": assistant_message,
            "message_id": str(result.inserted_id),
            "timestamp": assistant_chat_doc["created_at"].isoformat()
        }
        
    except Exception as e:
        logger.exception(f"Chat service failed for {symbol}")
        raise HTTPException(status_code=500, detail=str(e))

def get_user_signals_service(
    db,
    user_id: str,
    limit: int,
    symbol: Optional[str] = None,
    decision: Optional[str] = None
) -> List[Dict[str, Any]]:
    """Get user's latest trading signals"""
    
    try:
        query = {"user_id": user_id}
        
        if symbol:
            query["symbol"] = symbol.upper()
        
        if decision:
            query["analysis.decision"] = decision.upper()
        
        signals = list(db["analyses"].find(query).sort("timestamp", -1).limit(limit))
        
        # Format results
        formatted_signals = []
        for signal in signals:
            analysis = signal.get("analysis", {})
            formatted_signal = {
                "id": str(signal["_id"]),
                "symbol": signal["symbol"],
                "decision": analysis.get("decision", "HOLD"),
                "confidence": analysis.get("confidence", "LOW"),
                "entry_price": analysis.get("entry_price"),
                "stop_loss": analysis.get("stop_loss"),
                "targets": analysis.get("targets", []),
                "rationale": analysis.get("rationale", []),
                "technical_indicators": analysis.get("technical_indicators", {}),
                "risk_reward_ratio": analysis.get("risk_reward_ratio"),
                "timestamp": signal["timestamp"].isoformat(),
                "context": signal.get("context", "general")
            }
            formatted_signals.append(formatted_signal)
        
        return formatted_signals
        
    except Exception as e:
        logger.exception("Failed to get user signals")
        raise HTTPException(status_code=500, detail=str(e))

# ============ ORDER MANAGEMENT ============

def place_order_service(
    db,
    zerodha_client: ZerodhaClient,
    user_id: str,
    symbol: str,
    action: str,
    quantity: int,
    order_type: str,
    price: Optional[float] = None,
    analysis_id: Optional[str] = None,
    confirmed: bool = False
) -> Dict[str, Any]:
    """Place trading order with ChatGPT validation"""
    
    if not confirmed:
        raise HTTPException(status_code=400, detail="Order confirmation required")
    
    # Validate inputs
    if action not in ["BUY", "SELL"]:
        raise HTTPException(status_code=400, detail="Action must be BUY or SELL")
    
    if quantity <= 0:
        raise HTTPException(status_code=400, detail="Quantity must be positive")
    
    if order_type == "LIMIT" and not price:
        raise HTTPException(status_code=400, detail="Price required for LIMIT orders")
    
    # Check position size limits
    estimated_value = (price or 0) * quantity
    if estimated_value > MAX_POSITION_SIZE:
        raise HTTPException(status_code=400, detail=f"Position size exceeds limit of ₹{MAX_POSITION_SIZE}")
    
    try:
        # Get current market price for validation
        quote_data = zerodha_client.get_quote([f"NSE:{symbol}"])
        current_price = quote_data.get(f"NSE:{symbol}", {}).get("last_price", 0)
        
        if not current_price:
            raise HTTPException(status_code=400, detail="Unable to get current market price")
        
        # Price validation for LIMIT orders
        if order_type == "LIMIT" and price:
            price_diff_pct = abs(price - current_price) / current_price * 100
            if price_diff_pct > 10:  # 10% limit
                raise HTTPException(status_code=400, detail="LIMIT price too far from market price")
        
        # Prepare order payload
        order_payload = {
            "tradingsymbol": symbol,
            "exchange": "NSE",
            "transaction_type": action,
            "quantity": quantity,
            "product": "MIS",  # Intraday
            "order_type": order_type,
            "validity": "DAY"
        }
        
        if order_type == "LIMIT" and price:
            order_payload["price"] = price
        
        # Place order through Zerodha
        try:
            zerodha_response = zerodha_client.place_order(**order_payload)
            order_id = zerodha_response if isinstance(zerodha_response, str) else zerodha_response.get("order_id")
            
        except Exception as e:
            logger.exception(f"Zerodha order placement failed for {symbol}")
            raise HTTPException(status_code=502, detail=f"Order placement failed: {str(e)}")
        
        # Store order in database
        order_doc = {
            "user_id": user_id,
            "symbol": symbol,
            "action": action,
            "quantity": quantity,
            "order_type": order_type,
            "price": price,
            "current_market_price": current_price,
            "zerodha_order_id": order_id,
            "zerodha_response": zerodha_response,
            "order_payload": order_payload,
            "analysis_id": analysis_id,
            "status": "PLACED",
            "created_at": datetime.utcnow()
        }
        
        result = db["orders"].insert_one(order_doc)
        order_doc["_id"] = str(result.inserted_id)
        
        return {
            "order_id": str(result.inserted_id),
            "zerodha_order_id": order_id,
            "symbol": symbol,
            "action": action,
            "quantity": quantity,
            "price": price,
            "status": "PLACED",
            "timestamp": order_doc["created_at"].isoformat()
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.exception(f"Order service failed for {symbol}")
        raise HTTPException(status_code=500, detail=str(e))