# app/v1/services/gpt_prompts.py
"""Prompt templates and builders for GPT-based trading analysis.

This module is intentionally focused only on prompt construction so
that you can grow and tweak prompts independently of the GPT client
logic in `gpt_engine.py`.
"""

from typing import Dict, Any


def prepare_market_data_prompt(symbol: str, market_data: Dict[str, Any], question: str, context: str) -> str:
    """Build a rich, strictly-structured prompt for intraday analysis.

    We push *all* heavy lifting to GPT here: trend reading, early
    breakouts, pseudo-indicators, scoring and the final
    BUY/SELL/HOLD decision. Downstream code just consumes the JSON.

    The "context" parameter selects the ACTIVE STRATEGY for this
    request. For now we primarily support intraday SHORT_SELL and
    LONG_BUY daily trades, but the structure is future-proof for
    additional strategies.
    """

    # Map context to a single explicit active strategy + hint
    if context == "short_sell":
        strategy = "SHORT_SELL"
        strategy_hint = (
            "You are mainly looking for high-quality INTRADAY SHORT setups. "
            "Prefer SELL when there is strong, objective evidence for a short. "
            "However, you MUST still choose BUY or HOLD if price action is "
            "clearly bullish or no clean short edge exists."
        )
    elif context == "long_buy":
        strategy = "LONG_BUY"
        strategy_hint = (
            "You are mainly looking for high-quality INTRADAY LONG setups. "
            "Prefer BUY when there is strong, objective evidence for a long. "
            "However, you MUST still choose SELL or HOLD if conditions are "
            "clearly bearish or there is no clean long edge."
        )
    else:
        strategy = "NEUTRAL"
        strategy_hint = (
            "No specific directional bias. Choose BUY (long), SELL (short) "
            "or HOLD (no trade) purely from the data."
        )

    user_question = (question or "Provide intraday trading analysis").strip()

    prompt = f"""You are an expert discretionary intraday stock trader and risk manager.
You analyze order-flow, price action and multi-timeframe structure to
decide realistic intraday trades.

SYMBOL: {symbol}
ACTIVE STRATEGY: {strategy}
STRATEGY HINT: {strategy_hint}
PRIMARY USER QUESTION: {user_question}

DECISION SEMANTICS (STRICT):
- BUY  = Open / add LONG intraday position (direction up).
- SELL = Open SHORT intraday position (direction down). You may also
    implicitly exit any hypothetical long when recommending a new short.
- HOLD = No trade / stay flat (edge not strong enough yet).

YOUR CORE JOB:
1. Infer as much as possible from the raw candles and quote.
2. Score the quality of a potential intraday trade from 0-100.
3. Detect any EARLY BREAKOUT or strong momentum move.
4. Provide a machine-readable JSON decision including entry, stop,
     targets, rationale and a numeric score.

---
RAW MARKET DATA SNAPSHOT
"""

    # Add quote data
    quote = market_data.get("quote", {})
    if quote:
        prompt += f"""
CURRENT QUOTE (may be delayed slightly):
- Last Price: ₹{quote.get('last_price', 'N/A')}
- Open: ₹{quote.get('ohlc', {}).get('open', 'N/A')}
- High: ₹{quote.get('ohlc', {}).get('high', 'N/A')}
- Low: ₹{quote.get('ohlc', {}).get('low', 'N/A')}
- Close (prev): ₹{quote.get('ohlc', {}).get('close', 'N/A')}
- Volume: {quote.get('volume', 'N/A')}
- Change: {quote.get('net_change', 'N/A')} ({quote.get('net_change_percentage', 'N/A')}%)
"""

    # Add candle data summary per timeframe
    candles = market_data.get("candles", {})
    for timeframe, candle_data in candles.items():
        if candle_data and len(candle_data) > 0:
            recent_candles = candle_data[-10:]  # Last 10 candles per timeframe
            prompt += f"""
TIMEFRAME: {timeframe.upper()}  (Last 10 candles)
Format per line: index: O,H,L,C,Vol
"""
            for i, candle in enumerate(recent_candles):
                o = candle.get("open")
                h = candle.get("high")
                l = candle.get("low")
                c = candle.get("close")
                v = candle.get("volume", 0)
                prompt += f"  {i+1}: {o}, {h}, {l}, {c}, {v}\n"

    # Add numeric indicator summaries if available
    indicators = market_data.get("indicators", {})
    if indicators:
        prompt += """
---
INDICATOR SNAPSHOT (per timeframe, latest bar)
"""
        for timeframe, summary in indicators.items():
            ema_block = ""
            ema_values = summary.get("ema", {}) or {}
            if ema_values:
                # Compact EMA listing sorted by period
                ema_items = [f"EMA{p}={v}" for p, v in sorted(ema_values.items(), key=lambda x: int(x[0]))]
                ema_block = ", ".join(ema_items)

            prompt += f"""
TIMEFRAME: {timeframe.upper()} INDICATORS
- Close: {summary.get('close')}
- VWAP: {summary.get('vwap')}
- RSI: {summary.get('rsi')}
- MACD: {summary.get('macd')} / Signal: {summary.get('macd_signal')} / Hist: {summary.get('macd_hist')}
- Volume: {summary.get('volume')} (Δ {summary.get('volume_delta')})
- EMAs: {ema_block}
"""

    # Add pivot levels if present (currently daily)
    pivots = market_data.get("pivots", {})
    if pivots:
        prompt += """
---
PIVOT LEVELS (supports/resistances from higher timeframe)
"""
        for tf, levels in pivots.items():
            if not levels:
                continue
            prompt += f"""
TIMEFRAME: {tf.upper()} PIVOTS
- P: {levels.get('P')}
- R1/R2/R3: {levels.get('R1')}, {levels.get('R2')}, {levels.get('R3')}
- S1/S2/S3: {levels.get('S1')}, {levels.get('S2')}, {levels.get('S3')}
"""

    # Add Fibonacci retracement / extension levels if present
    fib_data = market_data.get("fib", {})
    if fib_data:
        prompt += """
---
FIBONACCI LEVELS (retracements and extensions from recent swing)
"""
        for tf, fib in fib_data.items():
            if not fib:
                continue
            swing_high = fib.get("swing_high")
            swing_low = fib.get("swing_low")
            retr = fib.get("retracements", {}) or {}
            ext = fib.get("extensions", {}) or {}

            prompt += f"""
TIMEFRAME: {tf.upper()} FIBONACCI
- Swing High: {swing_high}
- Swing Low: {swing_low}
- Retracements: {retr}
- Extensions: {ext}
"""

    # Add numeric strategy features (computed server-side) if present
    strategies = market_data.get("strategies", {})
    if strategies:
        per_tf = strategies.get("per_timeframe", {}) or {}
        mt = strategies.get("multi_timeframe", {}) or {}

        prompt += """
---
STRATEGY FEATURES (pre-computed, per timeframe)
These are numeric/enum flags only; do NOT invent new ones.
"""

        for timeframe, feats in per_tf.items():
            trend = feats.get("trend", {}) or {}
            vol = feats.get("volatility", {}) or {}
            rb = feats.get("range_breakout", {}) or {}
            orb = feats.get("opening_range", {}) or {}
            hod = feats.get("hod_lod", {}) or {}
            mb = feats.get("momentum_burst", {}) or {}
            vc = feats.get("volume_climax", {}) or {}
            pat = feats.get("pattern", {}) or {}
            fib_zone = feats.get("fib_zone", {}) or {}

            prompt += f"""
TIMEFRAME: {timeframe.upper()} STRATEGY FLAGS
- Trend: dir={trend.get('direction')} strength={trend.get('strength')}
- Volatility regime: {vol.get('regime')}
- Range breakout: status={rb.get('status')} high={rb.get('range_high')} low={rb.get('range_low')}
- Opening range: status={orb.get('status')}
- HOD/LOD proximity: near_hod={hod.get('near_hod')} near_lod={hod.get('near_lod')}
- Momentum burst: status={mb.get('status')} volume_ratio={mb.get('volume_ratio')}
- Volume climax: climax={vc.get('climax')} volume_ratio={vc.get('volume_ratio')}
- Candle pattern: {pat.get('pattern')}
- Fib zone (daily): {fib_zone.get('zone')}
"""

        if mt:
            prompt += f"""
MULTI-TIMEFRAME SUMMARY
- Trend alignment: {mt.get('trend_alignment')}
"""

    prompt += """
---
HOW TO THINK ABOUT THIS DATA (GUIDELINES):
- Use the candle sequences to approximate:
    * Trend direction (up / down / sideways) and strength.
    * Momentum (accelerating, slowing, reversing).
    * Support and resistance zones.
    * Volume expansion / exhaustion.
    * Early breakout or fake breakout attempts.
- Use shorter timeframes for fine entry/exit; longer ones for context
    if available.

SCORING & RANKING (VERY IMPORTANT):
- Create a numeric "score" from 0 to 100 where:
    * 0-30  = very weak / avoid trade.
    * 31-60 = medium quality / only trade with tight risk.
    * 61-80 = good quality setup.
    * 81-100 = exceptional / high-conviction setup.
- Score MUST be consistent with your BUY/SELL/HOLD decision.
    Example: HOLD should almost never have score > 40.

EARLY BREAKOUT & ACCUMULATION/DISTRIBUTION (APPROXIMATE):
- "early_breakout" should be TRUE only if:
    * Price is breaking a recent tight range or key level with
        expanding volume AND strong candles in one direction.
- "accumulation_distribution" summary should describe whether price
    action and volume look like accumulation, distribution or neither.

OUTPUT FORMAT (STRICT JSON ONLY):
Return a single JSON object with EXACTLY these keys (you can add
extra keys inside technical_indicators but keep the top-level keys).
{
    "decision": "BUY" | "SELL" | "HOLD",
    "confidence": "HIGH" | "MEDIUM" | "LOW",
    "entry_price": 1234.56,            // best approximate entry
    "stop_loss": 1200.00,              // protective stop level
    "targets": [1300.0, 1350.0],       // 1-3 reasonable intraday targets
    "score": 0,                        // integer 0-100 quality score
    "early_breakout": false,           // true if strong early breakout
    "early_breakout_comment": "",
    "rationale": [                     // 3-7 short bullet reasons
        "...",
        "..."
    ],
    "technical_indicators": {
        "trend": "bullish" | "bearish" | "sideways",
        "momentum": "strong" | "weak" | "neutral",
        "volume": "high" | "normal" | "low",
        "support": 0.0,                  // key nearby support
        "resistance": 0.0,               // key nearby resistance
        "accumulation_distribution": "accumulation" | "distribution" | "neutral",
        "pattern_summary": "short text about any chart/candle patterns detected"
    },
    "risk_reward_ratio": 2.5,
    "time_horizon": "intraday",
    "notes": "Any extra comments or warnings"
}

RULES:
- Respond **ONLY** with valid JSON that can be parsed by a machine.
- Do NOT wrap the JSON in code blocks.
- Do NOT add any commentary outside the JSON.
- You must output exactly one final "decision" (BUY/SELL/HOLD) and
    must NOT describe multiple alternative trades or separate options.
"""

    return prompt
