# app/v1/services/gpt_engine.py
"""
GPT-specific configuration and helper functions for the ChatGPT-based
trading engine. This module is responsible **only** for talking to
OpenAI models and building prompts; it has no Zerodha or DB logic.
"""

import os
import json
import logging
from typing import Dict, Any

from fastapi import HTTPException
from app.v1.services.gpt_prompts import prepare_market_data_prompt

# OpenAI integration
try:
    from openai import OpenAI
    OPENAI_AVAILABLE = True
except ImportError:  # pragma: no cover - handled at runtime
    OpenAI = None  # type: ignore
    OPENAI_AVAILABLE = False

logger = logging.getLogger(__name__)

# OpenAI configuration
OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "")
OPENAI_MODEL = os.getenv("OPENAI_MODEL", "gpt-4o")
OPENAI_TEMPERATURE = float(os.getenv("OPENAI_TEMPERATURE", "0.1"))
MAX_TOKENS = int(os.getenv("MAX_TOKENS", "1000"))
REQUEST_TIMEOUT = int(os.getenv("REQUEST_TIMEOUT", "30"))


def validate_openai_setup() -> Dict[str, Any]:
    """Validate OpenAI configuration and basic connectivity.

    This is used by the `/health` endpoint to report whether GPT
    analysis is available.
    """
    if not OPENAI_AVAILABLE:
        return {"configured": False, "message": "OpenAI package not installed"}

    if not OPENAI_API_KEY:
        return {"configured": False, "message": "OPENAI_API_KEY not set"}

    try:
        client = OpenAI(api_key=OPENAI_API_KEY)
        # Lightweight ping to verify credentials/model
        client.chat.completions.create(
            model=OPENAI_MODEL,
            messages=[{"role": "user", "content": "Test"}],
            max_tokens=10,
        )
        return {
            "configured": True,
            "message": f"OpenAI configured with model {OPENAI_MODEL}",
        }
    except Exception as e:  # pragma: no cover - runtime check
        return {"configured": False, "message": f"OpenAI test failed: {str(e)}"}


def get_openai_client() -> OpenAI:
    """Return an authenticated OpenAI client instance or raise HTTP error."""
    if not OPENAI_AVAILABLE:
        raise HTTPException(status_code=503, detail="OpenAI package not available")

    if not OPENAI_API_KEY:
        raise HTTPException(status_code=503, detail="OpenAI API key not configured")

    return OpenAI(api_key=OPENAI_API_KEY)


def call_chatgpt_analysis(prompt: str) -> Dict[str, Any]:
    """Call ChatGPT for structured JSON analysis with detailed logging."""
    try:
        logger.info(
            f"🤖 CHATGPT API CALL - Model: {OPENAI_MODEL}, Temp: {OPENAI_TEMPERATURE}"
        )
        client = get_openai_client()

        response = client.chat.completions.create(
            model=OPENAI_MODEL,
            messages=[
                {
                    "role": "system",
                    "content": (
                        "You are an expert intraday stock trader. "
                        "Always respond with valid JSON only."
                    ),
                },
                {"role": "user", "content": prompt},
            ],
            temperature=OPENAI_TEMPERATURE,
            max_tokens=MAX_TOKENS,
            timeout=REQUEST_TIMEOUT,
        )

        content = response.choices[0].message.content.strip()
        logger.info(f"📥 CHATGPT RAW RESPONSE (first 500 chars): {content[:500]}...")

        # Try to extract JSON from response
        try:
            start_idx = content.find("{")
            end_idx = content.rfind("}") + 1

            logger.info(f"🔍 JSON EXTRACTION - Start: {start_idx}, End: {end_idx}")

            if start_idx >= 0 and end_idx > start_idx:
                json_str = content[start_idx:end_idx]
                logger.info(f"📋 EXTRACTED JSON STRING: {json_str}")

                analysis = json.loads(json_str)
                logger.info("✅ JSON PARSED SUCCESSFULLY")
                logger.info(f"🎯 PARSED ANALYSIS KEYS: {list(analysis.keys())}")

                # Validate required fields
                required_fields = ["decision", "confidence", "rationale"]
                for field in required_fields:
                    if field not in analysis:
                        if field == "decision":
                            analysis[field] = "HOLD"
                        elif field == "confidence":
                            analysis[field] = "LOW"
                        else:
                            analysis[field] = ["No analysis provided"]

                return analysis

            logger.error(f"❌ NO JSON FOUND in response. Content: {content}")
            raise ValueError("No JSON found in response")

        except json.JSONDecodeError as e:
            logger.error(f"❌ JSON DECODE ERROR: {e}")
            logger.error(f"🔍 FAILED JSON STRING: {content[:1000]}")
            return {
                "decision": "HOLD",
                "confidence": "LOW",
                "rationale": [
                    "Analysis failed - invalid JSON response format",
                    f"Error: {str(e)}",
                ],
                "error": "JSON parse error",
                "raw_response": content[:500],
            }

    except Exception as e:  # pragma: no cover - external API
        logger.exception("ChatGPT analysis failed")
        return {
            "decision": "HOLD",
            "confidence": "LOW",
            "rationale": [f"Analysis failed: {str(e)}"],
            "error": str(e),
        }
