# app/v1/routers/teGPT.py
"""
Pure ChatGPT-based Trading Engine Router
No custom algorithms - all analysis delegated to OpenAI GPT models
"""
from fastapi import APIRouter, Depends, HTTPException, Body, Query
from typing import Dict, Any, List, Optional
import logging
from datetime import datetime
from bson import ObjectId

from app.db import database
from app.v1.dependencies.auth import get_current_userdetails
from app.v1.services.zerodha.client import ZerodhaClient
from app.v1.services.teGPT import (
    DEFAULT_TIMEFRAMES,
    get_zerodha_client_service,
    get_market_movers_service,
    analyze_symbol_service,
    bulk_analyze_service,
    chat_with_stock_service,
    place_order_service,
    get_user_signals_service,
    refresh_instruments_service,
    validate_openai_setup,
)

router = APIRouter()
logger = logging.getLogger(__name__)

# ============ HEALTH & SETUP ============

@router.get("/health", summary="Health check and OpenAI configuration status")
async def health_check():
    """Check system health and OpenAI configuration"""
    openai_status = validate_openai_setup()
    return {
        "status": "ok",
        "service": "ChatGPT Trading Engine",
        "openai_configured": openai_status["configured"],
        "openai_message": openai_status["message"]
    }

@router.get("/config", summary="Get current configuration")
async def get_config(current_user=Depends(get_current_userdetails)):
    """Get current system configuration"""
    return {
        "user_id": str(current_user.get("_id")),
        "model": "gpt-4o-mini",
        "features": {
            "bulk_analysis": True,
            "interactive_chat": True,
            "order_placement": True,
            "real_time_data": True
        }
    }

# ============ MARKET DATA ============

@router.get("/movers", summary="Get market top gainers/losers")
async def get_market_movers(
    type: str = Query("gainers", description="Type: gainers, losers, or both"),
    limit: int = Query(50, description="Maximum number of symbols"),
    db=Depends(database.get_mongo_db)
):
    """Get market movers (gainers/losers)"""
    try:
        result = get_market_movers_service(db, type, limit)
        return {"status": "success", "data": result}
    except Exception as e:
        logger.exception("Failed to get market movers")
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/movers/top10", summary="Get top 10 market movers (frontend compatibility)")
async def get_top10_movers(
    mover: str = Query("gainers", description="gainers or losers"),
    db=Depends(database.get_mongo_db)
):
    """Get top 10 market movers - compatible with existing frontend"""
    try:
        result = get_market_movers_service(db, mover, 10)
        symbols = result.get("symbols", [])[:10]
        
        return {
            "status": "ok",
            "top10": symbols,
            "source": result.get("source", "fallback"),
            "timestamp": result.get("timestamp")
        }
    except Exception as e:
        logger.exception("Failed to get top10 movers")
        raise HTTPException(status_code=500, detail=str(e))

@router.post("/instruments/refresh", summary="Refresh instrument data from Zerodha")
async def refresh_instruments(
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Refresh instrument master data from Zerodha"""
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        result = refresh_instruments_service(db, zerodha_client)
        return {"status": "success", "updated": result["count"], "timestamp": result["timestamp"]}
    except Exception as e:
        logger.exception("Failed to refresh instruments")
        raise HTTPException(status_code=500, detail=str(e))

# ============ ANALYSIS ============

@router.post("/analyze/{symbol}", summary="Analyze single symbol using ChatGPT")
async def analyze_symbol(
    symbol: str,
    payload: Dict[str, Any] = Body(default={}),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """
    Analyze a single symbol using ChatGPT
    Payload: {
        "timeframes": ["1minute", "5minute", "15minute"],
        "question": "Is this good for short selling?",
        "context": "intraday"
    }
    """
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        result = analyze_symbol_service(
            db=db,
            zerodha_client=zerodha_client,
            symbol=symbol,
            timeframes=payload.get("timeframes", DEFAULT_TIMEFRAMES),
            question=payload.get("question", "Provide intraday trading analysis"),
            context=payload.get("context", "intraday"),
            user_id=str(current_user.get("_id"))
        )
        return {"status": "success", "symbol": symbol, "analysis": result}
    except HTTPException:
        raise
    except Exception as e:
        logger.exception(f"Failed to analyze symbol {symbol}")
        raise HTTPException(status_code=500, detail=str(e))

@router.post("/analyze/bulk", summary="Bulk analyze multiple symbols")
async def bulk_analyze(
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """
    Bulk analyze multiple symbols
    Payload: {
        "symbols": ["RELIANCE", "TCS", "INFY"],
        "analysis_type": "short_sell" | "long_buy" | "general",
        "timeframes": ["5minute", "15minute"],
        "max_concurrent": 5
    }
    """
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        result = bulk_analyze_service(
            db=db,
            zerodha_client=zerodha_client,
            symbols=payload.get("symbols", []),
            analysis_type=payload.get("analysis_type", "general"),
            timeframes=payload.get("timeframes", DEFAULT_TIMEFRAMES),
            max_concurrent=payload.get("max_concurrent", 5),
            user_id=str(current_user.get("_id"))
        )
        return {"status": "success", "analyzed": len(result), "results": result}
    except Exception as e:
        logger.exception("Bulk analysis failed")
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/signals", summary="Get user's latest signals/analysis")
async def get_signals(
    limit: int = Query(20, description="Maximum number of signals"),
    symbol: Optional[str] = Query(None, description="Filter by symbol"),
    decision: Optional[str] = Query(None, description="Filter by decision: BUY, SELL, HOLD"),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Get user's latest trading signals from ChatGPT analysis"""
    try:
        result = get_user_signals_service(
            db=db,
            user_id=str(current_user.get("_id")),
            limit=limit,
            symbol=symbol,
            decision=decision
        )
        return {"status": "success", "signals": result}
    except Exception as e:
        logger.exception("Failed to get signals")
        raise HTTPException(status_code=500, detail=str(e))

# ============ INTERACTIVE CHAT ============

@router.post("/chat/{symbol}", summary="Interactive chat about a specific symbol")
async def chat_with_symbol(
    symbol: str,
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """
    Interactive chat about a symbol with real-time market data
    Payload: {
        "message": "What's the RSI looking like?",
        "include_fresh_data": true,
        "conversation_id": "optional-uuid"
    }
    """
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        result = chat_with_stock_service(
            db=db,
            zerodha_client=zerodha_client,
            symbol=symbol,
            message=payload.get("message", ""),
            user_id=str(current_user.get("_id")),
            conversation_id=payload.get("conversation_id"),
            include_fresh_data=payload.get("include_fresh_data", True)
        )
        return {"status": "success", "response": result}
    except Exception as e:
        logger.exception(f"Chat failed for symbol {symbol}")
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/chat/{symbol}/history", summary="Get chat history for a symbol")
async def get_chat_history(
    symbol: str,
    limit: int = Query(50, description="Maximum messages"),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Get chat history for a specific symbol"""
    try:
        user_id = str(current_user.get("_id"))
        chats = list(db["chats"].find({
            "user_id": user_id,
            "symbol": symbol
        }).sort("created_at", -1).limit(limit))
        
        for chat in chats:
            chat["_id"] = str(chat["_id"])
        
        return {"status": "success", "history": chats}
    except Exception as e:
        logger.exception(f"Failed to get chat history for {symbol}")
        raise HTTPException(status_code=500, detail=str(e))

# ============ ORDER MANAGEMENT ============

@router.post("/order/place", summary="Place order based on ChatGPT analysis")
async def place_trading_order(
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """
    Place trading order with ChatGPT risk validation
    Payload: {
        "symbol": "RELIANCE",
        "action": "BUY" | "SELL",
        "quantity": 10,
        "order_type": "MARKET" | "LIMIT",
        "price": 2500.0,  // Required for LIMIT orders
        "analysis_id": "optional-analysis-reference",
        "confirm": true  // Required for execution
    }
    """
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        result = place_order_service(
            db=db,
            zerodha_client=zerodha_client,
            user_id=str(current_user.get("_id")),
            symbol=payload.get("symbol"),
            action=payload.get("action"),
            quantity=payload.get("quantity"),
            order_type=payload.get("order_type", "MARKET"),
            price=payload.get("price"),
            analysis_id=payload.get("analysis_id"),
            confirmed=payload.get("confirm", False)
        )
        return {"status": "success", "order": result}
    except HTTPException:
        raise
    except Exception as e:
        logger.exception("Order placement failed")
        raise HTTPException(status_code=500, detail=str(e))

@router.get("/orders", summary="Get user's order history")
async def get_orders(
    limit: int = Query(50, description="Maximum orders"),
    status: Optional[str] = Query(None, description="Filter by status"),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Get user's order history"""
    try:
        user_id = str(current_user.get("_id"))
        query = {"user_id": user_id}
        if status:
            query["status"] = status
            
        orders = list(db["orders"].find(query).sort("created_at", -1).limit(limit))
        
        for order in orders:
            order["_id"] = str(order["_id"])
        
        return {"status": "success", "orders": orders}
    except Exception as e:
        logger.exception("Failed to get orders")
        raise HTTPException(status_code=500, detail=str(e))

# ============ LEGACY COMPATIBILITY ============

@router.get("/live", summary="Legacy compatibility - get live analysis")
async def get_live_analysis(
    mover: str = Query("gainers", description="gainers or losers"),
    limit: int = Query(10, description="Number of symbols to analyze"),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Legacy endpoint for compatibility with existing frontend"""
    try:
        logger.info(f"🚀 LIVE ANALYSIS STARTED - Type: {mover}, Limit: {limit}")
        
        # Get movers
        movers_result = get_market_movers_service(db, mover, limit)
        symbols = movers_result.get("symbols", [])[:limit]
        
        logger.info(f"🎯 SYMBOLS TO ANALYZE: {symbols}")
        
        if not symbols:
            logger.warning("⚠️ NO SYMBOLS FOUND - Returning empty results")
            return {"status": "success", "scanned": 0, "results": []}
        
        # Bulk analyze using a single neutral strategy context
        zerodha_client = get_zerodha_client_service(db, current_user)

        analysis_type = "general"
        logger.info(f"🤖 STARTING CHATGPT BULK ANALYSIS - Type: {analysis_type}")

        results = bulk_analyze_service(
            db=db,
            zerodha_client=zerodha_client,
            symbols=symbols,
            analysis_type=analysis_type,
            timeframes=DEFAULT_TIMEFRAMES,
            max_concurrent=3,
            user_id=str(current_user.get("_id"))
        )
        
        # Format for legacy + new frontend
        formatted_results = []
        for result in results:
            raw_targets = result.get("targets")
            exit_target = None
            if isinstance(raw_targets, (list, tuple)) and raw_targets:
                exit_target = raw_targets[0]

            formatted_results.append({
                "symbol": result["symbol"],
                "rank": len(formatted_results) + 1,
                "decision": result.get("decision", "HOLD"),
                "confidence": result.get("confidence", "LOW"),
                "instrument_token": result.get("instrument_token"),
                # Core numeric fields from GPT analysis
                "entry_price": result.get("entry_price"),
                "stop_loss": result.get("stop_loss"),
                "price_target": result.get("price_target"),
                "risk_reward_ratio": result.get("risk_reward_ratio"),
                "current_price": result.get("current_price"),
                "metrics": {
                    "decision": result.get("decision", "HOLD"),
                    "confidence": result.get("confidence", "LOW"),
                    "volume_drop": 0.0,
                    "momentum_drop": 0.0,
                    "ema_gap": 0.0,
                    "rsi_divergence": False,
                    "macd_divergence": False,
                    "wedge": False,
                    "pivot_level": "",
                    "near_upper_circuit": False,
                },
                # Full GPT rationale kept only in popup on frontend
                "rationale": result.get("rationale", []),
                # Compact technical + strategy features for popup
                "features": result.get("features", {}),
                "technical_indicators": result.get("technical_indicators", {}),
                "targets": {
                    "entry": result.get("entry_price"),
                    "exit": exit_target
                },
                "timestamp": result.get("timestamp")
            })
        
        return {
            "status": "success",
            "scanned": len(symbols),
            "results": formatted_results
        }
        
    except Exception as e:
        logger.exception("Legacy live analysis failed")
        raise HTTPException(status_code=500, detail=str(e))

# ============ FRONTEND COMPATIBILITY ============

@router.get("/profile", summary="Get Zerodha profile status")
async def get_profile(
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Get Zerodha profile and authentication status"""
    try:
        zerodha_client = get_zerodha_client_service(db, current_user)
        
        # Try to get profile to verify connection
        try:
            profile = zerodha_client.profile()
            return {
                "status": "ok",
                "authenticated": True,
                "profile": profile
            }
        except Exception as e:
            return {
                "status": "error",
                "authenticated": False,
                "message": "Zerodha authentication required"
            }
    except HTTPException as e:
        if e.status_code == 404:
            return {"status": "error", "authenticated": False, "message": "Zerodha settings not found"}
        elif e.status_code == 403:
            return {"status": "error", "authenticated": False, "message": "Zerodha not authenticated"}
        raise
    except Exception as e:
        logger.exception("Profile check failed")
        return {"status": "error", "authenticated": False, "message": str(e)}

@router.post("/stream/feedback/{analysis_id}", summary="Provide feedback on analysis")
async def stream_feedback(
    analysis_id: str,
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Provide thumbs up/down feedback on analysis"""
    try:
        feedback = payload.get("feedback")  # "up" or "down"
        if feedback not in ("up", "down"):
            raise HTTPException(status_code=400, detail="feedback must be 'up' or 'down'")
        
        # Update analysis with feedback
        try:
            result = db["analyses"].update_one(
                {"_id": ObjectId(analysis_id), "user_id": str(current_user.get("_id"))},
                {"$set": {"user_feedback": feedback, "feedback_at": datetime.utcnow()}}
            )
            if result.matched_count == 0:
                raise HTTPException(status_code=404, detail="Analysis not found")
        except Exception:
            # Fallback: try updating by string ID
            result = db["analyses"].update_one(
                {"analysis_id": analysis_id, "user_id": str(current_user.get("_id"))},
                {"$set": {"user_feedback": feedback, "feedback_at": datetime.utcnow()}}
            )
            if result.matched_count == 0:
                raise HTTPException(status_code=404, detail="Analysis not found")
        
        return {"status": "ok", "feedback": feedback}
    except HTTPException:
        raise
    except Exception as e:
        logger.exception("Feedback update failed")
        raise HTTPException(status_code=500, detail=str(e))

@router.post("/callback", summary="Zerodha OAuth callback")
async def oauth_callback(
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db),
    current_user=Depends(get_current_userdetails)
):
    """Handle Zerodha OAuth callback"""
    try:
        request_token = payload.get("request_token")
        user_id = str(current_user.get("_id"))

        if not request_token:
            raise HTTPException(status_code=400, detail="Missing request token")

        settings = db["zerodha_settings"].find_one({"user_id": user_id})
        if not settings:
            raise HTTPException(status_code=404, detail="Zerodha settings not found")

        from app.v1.services.zerodha.client import ZerodhaClient
        client = ZerodhaClient(settings["api_key"], settings["api_secret"])
        session = client.generate_session(request_token)

        # Update settings with access token
        db["zerodha_settings"].update_one(
            {"user_id": user_id},
            {"$set": {
                "access_token": session["access_token"],
                "login_time": datetime.utcnow(),
                "status": "active"
            }}
        )

        return {"status": "connected", "message": "Zerodha authentication successful"}
    except HTTPException:
        raise
    except Exception as e:
        logger.exception("OAuth callback failed")
        raise HTTPException(status_code=500, detail=str(e))

@router.post("/postback", summary="Zerodha postback handler")
async def zerodha_postback(
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db)
):
    """Handle Zerodha postback notifications"""
    try:
        # Store postback
        postback_doc = {
            "source": "zerodha",
            "payload": payload,
            "received_at": datetime.utcnow()
        }
        db["zerodha_postbacks"].insert_one(postback_doc)
        
        # Update order status if applicable
        order_id = payload.get("order_id")
        if order_id:
            db["orders"].update_one(
                {"zerodha_order_id": order_id},
                {"$set": {
                    "status": payload.get("status"),
                    "filled_quantity": payload.get("filled_quantity", 0),
                    "average_price": payload.get("average_price"),
                    "updated_at": datetime.utcnow()
                }}
            )
        
        return {"status": "received"}
    except Exception as e:
        logger.exception("Postback processing failed")
        raise HTTPException(status_code=500, detail=str(e))

# ============ WEBHOOKS ============

@router.post("/webhook/zerodha", summary="Zerodha order update webhook")
async def zerodha_webhook(
    payload: Dict[str, Any] = Body(...),
    db=Depends(database.get_mongo_db)
):
    """Handle Zerodha order update webhooks"""
    try:
        # Store webhook payload
        webhook_doc = {
            "source": "zerodha",
            "payload": payload,
            "received_at": datetime.utcnow()
        }
        db["webhooks"].insert_one(webhook_doc)
        
        # Update order status if applicable
        order_id = payload.get("order_id")
        if order_id:
            db["orders"].update_one(
                {"zerodha_order_id": order_id},
                {"$set": {
                    "status": payload.get("status"),
                    "filled_quantity": payload.get("filled_quantity", 0),
                    "average_price": payload.get("average_price"),
                    "updated_at": datetime.utcnow()
                }}
            )
        
        return {"status": "received"}
    except Exception as e:
        logger.exception("Webhook processing failed")
        raise HTTPException(status_code=500, detail=str(e))