# app/v1/routers/saas/zerodha.py

from fastapi import APIRouter, HTTPException, Depends, Query
from typing import Optional
from bson import ObjectId
from app.db import database
from app.v1.models.saas.zerodha import (
    ZerodhaSettingsCreate,
    ZerodhaSettingsUpdate,
    ZerodhaSettingsOut,
)

router = APIRouter()
COLLECTION_NAME = "zerodha_settings"

@router.get("/", response_model=Optional[ZerodhaSettingsOut])
def get_zerodha_settings(user_id: str = Query(...), db=Depends(database.get_mongo_db)):
    record = db[COLLECTION_NAME].find_one({"user_id": user_id})
    if record:
        record["_id"] = str(record["_id"])
        return ZerodhaSettingsOut(**record)
    return None

@router.post("/", response_model=ZerodhaSettingsOut)
def create_zerodha_settings(settings: ZerodhaSettingsCreate, db=Depends(database.get_mongo_db)):
    if db[COLLECTION_NAME].find_one({"user_id": settings.user_id}):
        raise HTTPException(status_code=400, detail="Settings already exist. Use PUT to update.")
    
    data = settings.dict()
    result = db[COLLECTION_NAME].insert_one(data)
    data["_id"] = str(result.inserted_id)
    return ZerodhaSettingsOut(**data)

@router.put("/{setting_id}", response_model=ZerodhaSettingsOut)
def update_zerodha_settings(setting_id: str, settings: ZerodhaSettingsUpdate, db=Depends(database.get_mongo_db)):
    try:
        object_id = ObjectId(setting_id)
    except Exception:
        raise HTTPException(status_code=400, detail="Invalid setting_id")

    update_data = {k: v for k, v in settings.dict().items() if v is not None}

    # 👇 Check if api_key or api_secret is being changed
    existing = db[COLLECTION_NAME].find_one({"_id": object_id})
    if not existing:
        raise HTTPException(status_code=404, detail="Settings not found")

    if (
        ('api_key' in update_data and update_data['api_key'] != existing.get('api_key')) or
        ('api_secret' in update_data and update_data['api_secret'] != existing.get('api_secret'))
    ):
        # Reset access token if auth config changes
        update_data['access_token'] = None
        update_data['public_token'] = None  # Optional: if you store it

    result = db[COLLECTION_NAME].update_one({"_id": object_id}, {"$set": update_data})

    updated = db[COLLECTION_NAME].find_one({"_id": object_id})
    updated["_id"] = str(updated["_id"])
    return ZerodhaSettingsOut(**updated)
