from pydantic import BaseModel, Field, EmailStr
from bson import ObjectId
from datetime import datetime
from typing import Optional, List

class UserBase(BaseModel):
    email: EmailStr
    password: str
    verificationCode: Optional[int] = None

    # Optional fields
    name: Optional[str] = ""
    mobile: Optional[str] = ""
    account_id: Optional[str] = ""
    created_date: Optional[datetime] = Field(default_factory=datetime.utcnow)
    last_login: Optional[datetime] = Field(default_factory=datetime.utcnow)
    date_of_birth: Optional[datetime] = None
    google_connect: Optional[str] = ""
    linkedin_connect: Optional[str] = ""
    token: Optional[str] = ""
    picurl: Optional[str] = ""

    # Default fields
    is_active: bool = True
    is_verified: bool = True
    mobile_verified: bool = False
    roles: int = 3

class User(UserBase):
    #id: str = Field(default=None, alias="_id")
    id: str
    password: Optional[str] = None

    class Config:
        arbitrary_types_allowed = True
        json_encoders = {ObjectId: str}
        allow_population_by_field_name = True


class UserResponse(BaseModel):
    user_id: str
    email: EmailStr
    verificationCode: Optional[int] = None
    name: Optional[str] = None
    mobile: Optional[str] = None
    account_id: Optional[str] = None
    created_date: Optional[datetime] = None
    last_login: Optional[datetime] = None
    date_of_birth: Optional[datetime] = None
    google_connect: Optional[str] = None
    linkedin_connect: Optional[str] = None    
    token: Optional[str] = ""
    picurl: Optional[str] = ""

    is_active: bool = True
    is_verified: bool = True
    mobile_verified: bool = False
    roles: int = 3

class UserResponseList(BaseModel):  
    total_count: int
    users: List[UserResponse]


class UserUpdate(BaseModel):
    account_id: Optional[str]
    name: Optional[str]
    mobile: Optional[str]
    profile_image: Optional[str] = None
    date_of_birth: Optional[datetime] = datetime.utcnow()
    is_active: Optional[bool] = True
    is_verified: Optional[bool] = True
    roles: Optional[int] = 3
    permissions: Optional[List[str]] = None


class UserCreate(UserBase):
    password: Optional[str]


class UserInDB(UserBase):
    hashed_password: str

class ResetPasswordRequest(BaseModel):
    email: str
    verificationCode: str
    password: str
    mode: str  # Include this if you need it, based on your frontend code


class PasswordReset(BaseModel):
    email: EmailStr


class PasswordChange(BaseModel):
    current_password: str
    new_password: str


class Token(BaseModel):
    access_token: str
    token_type: str


class TokenData(BaseModel):
    email: str

class EmailAction(BaseModel):
    emails: str
    action: Optional[str] = 'signup'  # 'signup', 'invite', 'forgot'
    account_id: Optional[str] = None  # Required for 'invite' action
    role: int = 4

class GoogleToken(BaseModel):
    token: str

class UserLogin(BaseModel):
    email: EmailStr
    password: str

