from pydantic import BaseModel
from typing import Optional
from enum import Enum


class Address(BaseModel):
    street: Optional[str] = None
    city: Optional[str] = None
    state: Optional[str] = None
    country: Optional[str] = None
    postal_code: Optional[str] = None

class AccountType(int, Enum):
    MAIN = 1
    PARTNER = 3
    ACCOUNT = 100

class SubscriptionType(str, Enum):
    FREE = "Free"               # Free plan with limited features, supports up to 5 users
    PREMIUM = "Premium"         # Paid plan with limited features (e.g., $10/user)
    ENTERPRISE = "Enterprise"   # Full-featured paid plan (e.g., $20/user)

class SubscriptionStatus(str, Enum):
    trial = "Trial"             # Currently in trial period with active access
    active = "Active"           # Subscription is paid and active
    unpaid = "Unpaid"           # Payment pending or failed, access may still be active temporarily
    expired = "Expired"         # Subscription duration has ended, access is no longer active
    cancelled = "Cancelled"     # Subscription cancelled by the user
    suspended = "Suspended"     # Temporarily disabled by system/admin (e.g., violation or manual action)
    paused = "Paused"           # Temporarily paused by user or admin (optional state)

class InvoiceStatus(str, Enum):
    NEW = "New"               # Invoice has been fully paid
    PAID = "Paid"               # Invoice has been fully paid
    UNPAID = "Unpaid"           # Invoice is pending payment
    OVERDUE = "Overdue"         # Payment is past due date and still unpaid
    CANCELLED = "Cancelled"     # Invoice has been cancelled or voided

class InvoiceType(str, Enum):
    SETUP = "Setup"               # Invoice has been fully paid
    SUBSCRIPTION = "Subscription"           # Invoice is pending payment
    DEVELOPMENT = "Development"         # Payment is past due date and still unpaid
    SUPPORT = "Support"     # Invoice has been cancelled or voided
    MISC = "Misc"

class PaymentStatus(str, Enum):
    PAID = "Paid"               # Payment completed successfully
    DECLINED = "Declined"       # Payment was declined by the payment processor
    DISPUTED = "Disputed"       # Payment is under dispute (e.g., chargeback)
    HOLD = "Hold"               # Payment is on hold (e.g., manual review, processing delay)

class PaymentType(str, Enum):
    STRIPE = "STRIPE"           # Payment made via Stripe
    CC = "CC"                   # Payment via Credit Card
    CHECK = "Check/Cash"       # Payment made via cheque or cash
    WIRE = "WIRE"               # Payment made via bank wire transfer

class WorkforceCapability(str, Enum):
    TRACKING = "Tracking"
    SCHEDULING = "Scheduling & Dispatch"
    ROUTING = "Routing & Optimization"

class FleetCapability(str, Enum):
    TRACKING = "Tracking"
    INTELLIGENCE = "Intelligence"
    MAINTENANCE = "Maintenance"
    SCHEDULING = "Scheduling & Dispatch"

class PurposeOfUse(str, Enum):
    MOVE_PEOPLE = "Move People"
    MOVE_GOODS = "Move Goods"
    MOVE_SERVICES = "Move Services (Workforce)"


class WorkforceCountRange(str, Enum):
    range_0_10 = "0-10"
    range_11_100 = "11-100"
    range_101_500 = "101-500"
    range_501_1000 = "501-1000"
    range_1000_plus = "1001-10000"
    range_10000_plus = "10000+"

class RevenueRange(str, Enum):
    range_0_1M = "0-1M"
    range_1_10M = "1-10M"
    range_10_100M = "10-100M"
    range_100_500M = "100-500M"
    range_500M_plus = "500M-1B"
    range_1B_100B = "1B-100B"
    range_100B_plus = "100B+"

class Industry(Enum):
    TRANSPORTATION_MOBILITY = "Transportation & Mobility"
    LOGISTICS_SUPPLY_CHAIN = "Logistics & Supply Chain"
    HEALTHCARE_MEDICAL = "Healthcare & Medical Services"
    EDUCATION_INSTITUTIONS = "Education & Institutions"
    GOVERNMENT_PUBLIC_SECTOR = "Government & Public Sector"
    TELECOMMUNICATIONS_BROADBAND = "Telecommunications & Broadband"
    OIL_GAS_ENERGY = "Oil, Gas & Energy"
    CONSTRUCTION_INFRASTRUCTURE = "Construction & Infrastructure"
    HOME_FACILITY_SERVICES = "Home & Facility Services"
    FIELD_ON_DEMAND_SERVICES = "Field & On-Demand Services"
    RETAIL_HYPERLOCAL_DELIVERY = "Retail & Hyperlocal Delivery"
    FOOD_BEVERAGE = "Food & Beverage"
    COURIER_EXPRESS_DELIVERY = "Courier & Express Delivery"
    WASTE_MANAGEMENT_RECYCLING = "Waste Management & Recycling"
    SECURITY_EMERGENCY_RESPONSE = "Security & Emergency Response"
    TOURISM_HOSPITALITY = "Tourism & Hospitality"
    EVENT_ENTERTAINMENT = "Event Management & Entertainment"
    RENTAL_LEASING = "Rental & Leasing Services"
    MANUFACTURING_INDUSTRIAL = "Manufacturing & Industrial Operations"
    AUTOMOTIVE_MAINTENANCE = "Automotive & Maintenance Services"
    PROFESSIONAL_WORKFORCE_SERVICES = "Professional & Workforce Services"
    AGRICULTURE_FARMING = "Agriculture & Farming"
    REAL_ESTATE_PROPERTY = "Real Estate & Property Management"
    BANKING_FINANCIAL = "Banking & Financial Services"
    UTILITIES_PUBLIC_WORKS = "Utilities & Public Works"
    OTHER = "Others"


