import asyncio
import logging
from datetime import datetime
from zoneinfo import ZoneInfo
from typing import Optional

from app.db.database import get_mongo_db
from app.v1.services.tradeGPT import refresh_movers_service

logger = logging.getLogger(__name__)

IST = ZoneInfo("Asia/Kolkata")


def _is_market_hours_ist(now_utc: Optional[datetime] = None) -> bool:
    """Return True if current time is within IST market hours.

    Market hours: Monday–Friday, 09:00–16:00 IST.
    """
    if now_utc is None:
        now_utc = datetime.utcnow()

    now_ist = now_utc.replace(tzinfo=ZoneInfo("UTC")).astimezone(IST)

    # Monday=0, Sunday=6
    if now_ist.weekday() > 4:
        return False

    hour = now_ist.hour
    minute = now_ist.minute

    # Simple range: 09:00 <= time < 16:00
    return (hour > 9 or (hour == 9 and minute >= 0)) and (hour < 16)


async def _refresh_movers_once() -> None:
    """Run a single ET movers refresh cycle using the shared DB helper.

    This reuses the existing `refresh_movers_service` so all logic
    (ET scraping, symbol mapping, and `movers` collection updates)
    stays in one place.
    """
    db_gen = get_mongo_db()
    db = next(db_gen)
    try:
        record = refresh_movers_service(db)
        gainers = len(record.get("gainers", []))
        losers = len(record.get("losers", []))
        logger.info(
            "[MoversRefresh] Refreshed movers: %s | gainers=%d losers=%d",
            record.get("fetched_at"),
            gainers,
            losers,
        )
    except Exception:  # pragma: no cover - defensive logging
        logger.exception("[MoversRefresh] Error refreshing ET movers")
    finally:
        try:
            db_gen.close()
        except Exception:
            # If closing fails, just log and move on; connection will be GC'ed.
            logger.debug("[MoversRefresh] Error closing DB generator", exc_info=True)


async def movers_refresh_loop(interval_seconds: int = 900) -> None:
    """Background loop to refresh ET movers periodically.

    Intended to be started from FastAPI's startup event via
    `asyncio.create_task(movers_refresh_loop(...))`.

    Parameters
    ----------
    interval_seconds: int
        Sleep interval between refresh cycles. Default: 900s (15 minutes).
    """
    # Small initial delay so the app can finish startup cleanly.
    await asyncio.sleep(5)
    logger.info(
        "[MoversRefresh] Background ET movers loop started (interval=%ss)",
        interval_seconds,
    )
    while True:
        if _is_market_hours_ist():
            await _refresh_movers_once()
        else:
            logger.info("[MoversRefresh] Outside IST market hours; skipping refresh")
        await asyncio.sleep(interval_seconds)
